/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.api;

import java.util.Collection;
import java.util.Map;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;

import com.xebialabs.xlplatform.documentation.PublicApi;

/**
 * Manages permissions in XL Deploy.
 */
@Path("/security")
@Consumes({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@Produces({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@PublicApi
public interface PermissionService {

    /**
     * Gets all the permissions granted to a role.
     *
     * @permission security#edit
     * @param role the role to get the permissions for.
     * @return a Map of configuration item ids to permissions granted to the user.
     */
    @GET
    @Path("granted-permissions/{role}")
    Map<String, Collection<String>> getGrantedPermissions(@PathParam("role") String role);

    /**
     * Gets all the permissions granted to a role.
     *
     * @permission security#edit
     * @param roleId the role to get the permissions for.
     * @return a Map of configuration item ids to permissions granted to the user.
     */
    @GET
    @Path("granted-permissions/id/{roleId}")
    Map<String, Collection<String>> getGrantedPermissionsById(@PathParam("roleId") String roleId);

    /**
     * Gets all the permissions granted to the logged in user.
     *
     * @return a Map of configuration item ids to permissions granted the currently logged in user.
     */
    @GET
    @Path("granted-permissions")
    Map<String, Collection<String>> getMyGrantedPermissions();

    /**
     * Checks if a permission is granted to a role on a CI.
     *
     * @permission security#edit
     * @param permission the name of the permission to check.
     * @param id the path of the CI to check the permission on.
     * @param role the role to which the permission should be granted.
     * @return {@code true} if granted.
     */
    @GET
    @Path("permission/{permission}/{role}/{id:.+}")
    boolean isGranted(@PathParam("permission") String permission, @PathParam("id") String id, @PathParam("role") String role);

    /**
     * Checks if a permission is granted to a role on a CI.
     *
     * @permission security#edit
     * @param permission the name of the permission to check.
     * @param role the role to which the permission should be granted.
     * @return {@code true} if granted.
     */
    @GET
    @Path("permission/{permission}/{role}")
    default boolean isGranted(@PathParam("permission") String permission, @PathParam("role") String role) {
        return isGranted(permission, "", role);
    }

    /**
     * Grants a permission to a role on a CI.
     *
     * @permission security#edit
     * @param permission the name of the permission to grant.
     * @param id the path of the CI to grant the permission on.
     * @param role the role to which the permission should be granted.
     */
    @PUT
    @Path("permission/{permission}/{role}/{id:.*}")
    void grant(@PathParam("permission") String permission, @PathParam("id") String id, @PathParam("role") String role);

    /**
     * Grants a permission to a role on a CI.
     *
     * @permission security#edit
     * @param permission the name of the permission to grant.
     * @param role the role to which the permission should be granted.
     */
    @PUT
    @Path("permission/{permission}/{role}")
    default void grant(@PathParam("permission") String permission, @PathParam("role") String role) {
        grant(permission, "", role);
    }

    /**
     * Revokes the permission of a role on a CI.
     *
     *
     * @permission security#edit
     * @param permission the name of the permission to revoke.
     * @param id the path of the CI to revoke the permission from.
     * @param role the role from which the permission should be revoked.
     */
    @DELETE
    @Path("permission/{permission}/{role}/{id:.+}")
    void revoke(@PathParam("permission") String permission, @PathParam("id") String id, @PathParam("role") String role);

    /**
     * Revokes the permission of a role on a CI.
     *
     *
     * @permission security#edit
     * @param permission the name of the permission to revoke.
     * @param role the role from which the permission should be revoked.
     */
    @DELETE
    @Path("permission/{permission}/{role}")
    default void revoke(@PathParam("permission") String permission, @PathParam("role") String role) {
        revoke(permission, "", role);
    }

    /**
     * Checks if the currently logged in user has a certain permission on a CI.
     *
     * @param permission the name of the permission to check.
     * @param id the path of the CI to check the permission on.
     * @return {@code true} if the user has this permission; {@code false} otherwise.
     */
    @GET
    @Path("check/{permission}/{id:.+}")
    boolean isGrantedToMe(@PathParam("permission") String permission, @PathParam("id") String id);

    /**
     * Checks if the currently logged in user has a certain permission on a CI.
     *
     * @param permission the name of the permission to check.
     * @return {@code true} if the user has this permission; {@code false} otherwise.
     */
    @GET
    @Path("check/{permission}")
    default boolean isGrantedToMe(@PathParam("permission") String permission) {
        return isGrantedToMe(permission, "");
    }
}
