/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.api;

import com.xebialabs.deployit.engine.api.dto.Ordering;
import com.xebialabs.deployit.engine.api.dto.Paging;
import com.xebialabs.deployit.engine.api.security.Role;
import com.xebialabs.deployit.engine.api.security.RolePrincipals;
import com.xebialabs.xlplatform.documentation.PublicApi;

import java.util.List;

import javax.ws.rs.BeanParam;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

/**
 * Manages the roles in XL Deploy's security system.
 */
@Path("/security/role")
@Consumes({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
@PublicApi
public interface RoleService {

	/**
	 * Checks if role with given name exists
	 *
	 * @param roleName checked role name
	 * @return true or false
	 */
	@GET
	@Path("exists/{name}")
	boolean exists(@PathParam("name") String roleName);

	/**
	 * List the roles for given configuration item ID
	 *
	 * @param id configuration item ID
	 * @param rolePattern the role name pattern used for search
	 * @param paging search paging
	 * @param order search order
	 * @return a list of roles.
	 */
	@GET
	@Path("v2/{id:.+}")
	List<Role> list(@PathParam("id") String id,
			@QueryParam("rolePattern") String rolePattern,
			@BeanParam Paging paging,
			@QueryParam("order") Ordering order);


	/**
     * Lists the names of all available roles in the security system.
     *
     * @return a list of role names.
     */
    @GET
    @Path("v2")
    List<String> list(@QueryParam("rolePattern") String rolePattern,
                      @BeanParam Paging paging,
                      @QueryParam("order") Ordering order);

    /**
     * Lists the names of all available roles in the security system.
     *
     * @return a list of role names.
     */
    @GET
    default List<String> list() {
        return this.list(null, null, null);
    }

    /**
     * Lists the roles of the currently logged in user.
     *
     * @return a list of role names.
     */
    @GET
    @Path("v2/roles")
    List<String> listMyRoles(@QueryParam("rolePattern") String rolePattern,
                             @BeanParam Paging paging,
                             @QueryParam("order") Ordering order);

    /**
     * Lists the roles of the currently logged in user.
     *
     * @return a list of role names.
     */
    @GET
    @Path("roles")
    default List<String> listMyRoles() {
        return this.listMyRoles(null, null, null);
    }

    /**
     * Lists the roles of a user.
     *
     * @param username the username of the principal
     * @return a list of role names.
     * @permission security#edit
     */
    @GET
    @Path("v2/roles/{username}")
    List<String> listRoles(@PathParam("username") String username,
                           @QueryParam("rolePattern") String rolePattern,
                           @BeanParam Paging paging,
                           @QueryParam("order") Ordering order);

    /**
     * Lists the roles of a user.
     *
     * @param username the username of the principal
     * @return a list of role names.
     * @permission security#edit
     */
    @GET
    @Path("roles/{username}")
    default List<String> listRoles(@PathParam("username") String username) {
        return this.listRoles(username, null, null, null);
    }

    /**
     * Creates a new role. Does nothing if the role already exists.
     *
     * @param name the name of the new role.
     * @permission security#edit
     */
    @PUT
    @Path("{role}")
    void create(@PathParam("role") String name);

	/**
	 * Update principals for existing role
	 *
	 * @param rolePrincipals rolePrincipals.role for which the rolePrincipals.principals is applied to
	 */
	@POST
	@Path("principals")
	void updatePrincipals(RolePrincipals rolePrincipals);

	/**
	 * Assigns a role to a user or group. The role will be created if it doesn't exist yet.
	 *
	 * @param name       the name of the role to assign
	 * @param principals the user or group to assign the role to.
	 * @permission security#edit
	 */
	@POST
	@Path("{role}/principals")
	void updatePrincipals(@PathParam("role") String name, List<String> principals);

	/**
     * Assigns a role to a user or group. The role will be created if it doesn't exist yet.
     *
     * @param name      the name of the role to assign
     * @param principal the user or group to assign the role to.
     * @permission security#edit
     */
    @PUT
    @Path("{role}/{principal}")
    void assign(@PathParam("role") String name, @PathParam("principal") String principal);

    /**
     * Removes a role from a user or group.
     *
     * @param name      the name of the role to remove
     * @param principal the user or group to remove the role from.
     * @permission security#edit
     */
    @DELETE
    @Path("{role}/{principal}")
    void unassign(@PathParam("role") String name, @PathParam("principal") String principal);

    /**
     * Renames a role. Does nothing if the role does not exist.
     *
     * @param name    the current name
     * @param newName the new name
     * @permission security#edit
     */
    @POST
    @Path("{role}")
    void rename(@PathParam("role") String name, String newName);

    /**
     * Removes a role from the XL Deploy security system. All assignments to principals are deleted as well.
     *
     * @param name the name of the role to delete.
     * @permission security#edit
     */
    @DELETE
    @Path("{role}")
    void delete(@PathParam("role") String name);

	/**
	 * Lists paged role principals available in the security system for given role name pattern
	 *
	 * @param rolePattern the role name pattern used for search
	 * @param paging search paging
	 * @param order search order
	 */
	@GET
	@Path("v2/principals")
	List<RolePrincipals> readRolePrincipals(@QueryParam("rolePattern") String rolePattern,
			@BeanParam Paging paging,
			@QueryParam("order") Ordering order);
	/**
	 * Lists all role principals available in the security system.
	 *
	 */
	@GET
	@Path("principals")
	default List<RolePrincipals> readRolePrincipals() {
		return this.readRolePrincipals(null, null, null);
	}
}
