/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.api;

import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.engine.api.dto.*;
import com.xebialabs.deployit.engine.api.dto.report.AppEnvStatusReport;
import com.xebialabs.deployit.engine.api.dto.report.DeploymentsStatusOverview;
import com.xebialabs.deployit.engine.api.execution.FetchMode;
import com.xebialabs.deployit.engine.api.execution.TaskKind;
import com.xebialabs.deployit.engine.api.execution.TaskWithBlock;
import com.xebialabs.xlplatform.documentation.PublicApi;

import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import jakarta.xml.bind.annotation.XmlElementWrapper;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;


/**
 * Retrieves reports for Deployment or Control Tasks A User must have a REPORT_VIEW permission to be able to run these
 * operations. A User must have at least READ permission on the environment or application for Deployment Task report to
 * contain data for them. A User must have at least CONTROLTASK_EXECUTE permission on target CI for Control Task report
 * to contain data for them.
 */
@Path("/report")
@Consumes({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@Produces({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@PublicApi
public interface ReportService {

    /**
     * Find all Application CIs availble in reporting for given user This will include both names of existing CIs user has permissions for
     * and a list of CIs that are existing in reports but are deleted or renamed. Deleted CIs are visible only to user with ADMIN rights.
     *
     * @headers Accept application/json
     * @return a list of {@link com.xebialabs.deployit.engine.api.dto.ConfigurationItemId} objects
     */
    @GET
    @Path("applications")
    List<ConfigurationItemId> listApplications();

    /**
     * Find all archived Control Tasks, optionally filtered by parameters.
     *
     * @param users (Optional) filter tasks belonging to users
     * @param states (Optional) filter on tasks in states
     * @param taskName (Optional) filter on task name
     * @param begin (Optional) filter by time interval beginning with date
     * @param end (Optional) filter by time interval ending with date
     * @param paging Which page and how many results per page to retrieve
     * @param order Which ordering to use when listing results
     * @param targets filter by list of target CIs
     * @param workerName (Optional) filter only specified workers
     * @param fetchMode toggle to fetch FULL task with blocks or just task SUMMARY
     * @param taskId (Optional) filter by task id
     * @return a list of {@link ControlTaskReportLine} objects
     */
    @POST
    @Path("controltasks")
    @XmlElementWrapper(name = "report")
    Stream<ControlTaskReportLine> getControlTasksReport(
            @QueryParam("begin") Date begin,
            @QueryParam("end") Date end,
            @QueryParam("states") List<String> states,
            @QueryParam("taskName") String taskName,
            @QueryParam("fetchMode") FetchMode fetchMode,
            @QueryParam("users") List<String> users,
            @QueryParam("order") List<Ordering> order,
            @QueryParam("worker") String workerName,
            @QueryParam("taskId") String taskId,
            @BeanParam Paging paging,
            Set<String> targets);

    /**
     * Find all archived Deployment Tasks, optionally filtered by parameters.
     *
     * @headers Content-Type application/json
     * @headers Cookie &lt;Cookie Value&gt;
     * @headers X-XSRF-TOKEN &lt;X-XSRF-TOKEN&gt;
     * @param taskId (Optional) filter task with exact taskId
     * @param onlySuccessful Deprecated use taskStates instead
     * @param users (Optional) filter tasks belonging to users
     * @param taskStates (Optional) filter on tasks in states
     * @param taskTypes (Optional) filter on tasks types
     * @param begin filter by time interval beginning with date
     * @param end filter by time interval ending with date
     * @param paging (Optional) Which page and how many results per page to retrieve
     * @param order (Optional) Which ordering to use when listing results
     * @param fetchMode (Optional) toggle to fetch FULL task with blocks or just task SUMMARY
     * @param workerName (Optional) filter only specified workers
     * @param configurationItemIds filter by list of CIs - environment and application correlated to the Deployment task
     * @return a list of {@link DeploymentTaskReportLine} objects
     */
    @POST
    @Path("tasks")
    @XmlElementWrapper(name = "report")
    Stream<DeploymentTaskReportLine> getTaskReport(
            @QueryParam("begin") Date begin,
            @QueryParam("end") Date end,
            @BeanParam Paging paging,
            @QueryParam("order") List<Ordering> order,
            @QueryParam("users") List<String> users,
            @QueryParam("states") List<String> taskStates,
            @QueryParam("types") List<String> taskTypes,
            @QueryParam("fetchMode") FetchMode fetchMode,
            @QueryParam("taskId") String taskId,
            @QueryParam("onlySuccessful") boolean onlySuccessful,
            @QueryParam("worker") String workerName,
            List<ConfigurationItemId> configurationItemIds);

    /**
     * Find all archived Control Tasks, optionally filtered by parameters.
     *
     * @headers Content-Type application/json
     * @headers Cookie &lt;Cookie Value&gt;
     * @headers X-XSRF-TOKEN &lt;X-XSRF-TOKEN&gt;
     * @param users (Optional) filter tasks belonging to users
     * @param states (Optional) filter on tasks in states
     * @param taskName (Optional) filter on task name
     * @param begin (Optional) filter by time interval beginning with date
     * @param end (Optional) filter by time interval ending with date
     * @param order (Optional) Which ordering to use when listing results
     * @param workerName (Optional) filter only specified workers
     * @param taskId (Optional) filter by task id
     * @param targets filter by list of target CIs
     * @return a {java.util.String} containing CSV report
     */
    @POST
    @Path("download/controltasks")
    @XmlElementWrapper(name = "report")
    String downloadControlTasksReport(
            @QueryParam("begin") Date begin,
            @QueryParam("end") Date end,
            @QueryParam("states") List<String> states,
            @QueryParam("taskName") String taskName,
            @QueryParam("users") List<String> users,
            @QueryParam("order") List<Ordering> order,
            @QueryParam("worker") String workerName,
            @QueryParam("taskId") String taskId,
            Set<String> targets);

    /**
     * Find all archived Deployment Tasks, optionally filtered by parameters.
     *
     * @headers Content-Type application/json
     * @headers Cookie &lt;Cookie Value&gt;
     * @headers X-XSRF-TOKEN &lt;X-XSRF-TOKEN&gt;
     * @param begin filter by time interval beginning with date
     * @param end filter by time interval ending with date
     * @param order (Optional) Which ordering to use when listing results
     * @param users (Optional) filter tasks belonging to users
     * @param taskStates (Optional) filter on tasks in states
     * @param taskTypes (Optional) filter on tasks types
     * @param taskId (Optional) filter task with exact taskId
     * @param onlySuccessful Deprecated use taskStates instead
     * @param workerName (Optional) filter only specified workers
     * @param configurationItemIds filter by list of CIs - environment and application correlated to the Deployment task
     * @return a {java.util.String} containing CSV report
     */
    @POST
    @Path("download/tasks")
    String downloadTaskReport(
            @QueryParam("begin") Date begin,
            @QueryParam("end") Date end,
            @QueryParam("order") List<Ordering> order,
            @QueryParam("users") List<String> users,
            @QueryParam("states") List<String> taskStates,
            @QueryParam("types") List<String> taskTypes,
            @QueryParam("taskId") String taskId,
            @QueryParam("onlySuccessful") boolean onlySuccessful,
            @QueryParam("worker") String workerName,
            List<ConfigurationItemId> configurationItemIds);

    /**
     * Find all Environment CIs availble in reporting for given user. This will include both names of existing CIs user has permissions for
     * and a list of CIs that are existing in reports but are deleted or renamed. Deleted CIs are visible only to user with ADMIN rights.
     *
     * @headers Accept application/json
     * @return a list of {@link com.xebialabs.deployit.engine.api.dto.ConfigurationItemId} objects
     */
    @GET
    @Path("environments")
    List<ConfigurationItemId> listEnvironments();

    /**
     * Export audit report containing global and folder permissions of roles
     *
     * @return a {java.util.String} containing audit report
     */
    @POST
    @Path("download/audit")
    String downloadAuditReport();

    /**
     * Export audit report containing global and folder permissions of roles
     *
     * @param folders Which folders in audit report
     * @return a {java.util.String} containing audit report
     */
    @POST
    @Path("download/audit/v2")
    String downloadAuditReport(@QueryParam("folders") List<String> folders);

    /**
     * Generates preview audit report for local and global permissions.
     *
     * @param folders Which folders in audit report
     * @param order Which ordering to use when listing results
     * @param paging Which page and how many results per page to retrieve
     *
     * @return a list of {@link AuditPreviewRow} objects
     */
    @POST
    @Path("preview/audit")
    @XmlElementWrapper(name = "report")
    Stream<AuditPreviewRow> previewAuditReport(
            @QueryParam("folders") List<String> folders,
            @QueryParam("order") List<Ordering> order,
            @BeanParam Paging paging);

    /**
     * Fetch overall number of deployments per deployment state.
     *
     * @param begin filter by time interval beginning with date
     * @param end filter by time interval ending with date
     *
     * @return a {@link DeploymentsStatusOverview} object
     */
    @GET
    @Path("status-overview")
    DeploymentsStatusOverview getStatusOverview(@QueryParam("begin") Date begin, @QueryParam("end") Date end);

    /**
     * Fetch number of deployments per application and environment.
     *
     * @param begin filter by time interval beginning with date
     * @param end filter by time interval ending with date
     *
     * @return a list of {@link AppEnvStatusReport} objects
     */
    @GET
    @Path("app-env-status")
    List<AppEnvStatusReport> getAppEnvStatus(@QueryParam("begin") Date begin, @QueryParam("end") Date end);

    /**
     * Find all archived Tasks by archived date, optionally filtered by parameters.
     * @permission report#view
     * @param begin filter by time interval beginning with date
     * @param end filter by time interval ending with date
     * @param order (Optional) Which ordering to use on archived date when listing results
     * @param paging (Optional) Which page and how many results per page to retrieve
     * @param fetchMode (Optional) toggle to fetch FULL task with blocks or just task SUMMARY
     * @param taskType (Optional) Filter by task type. DEPLOYMENT or CONTROL_TASK
     * @return a list of tasks. response header X-Total-Count determines the total number of tasks.
     */
    @GET
    @Path("tasks/query-by-archived-date")
    Stream<TaskWithBlock> queryByArchivedDate(
            @QueryParam("begin") Date begin,
            @QueryParam("end") Date end,
            @BeanParam Paging paging,
            @QueryParam("order") String order,
            @QueryParam("fetchMode") FetchMode fetchMode,
            @QueryParam("taskType") TaskKind taskType);
}
