/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.api.dto;

import com.xebialabs.xlplatform.documentation.PublicApiRef;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;

import jakarta.xml.bind.annotation.XmlElement;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.XmlTransient;

import com.xebialabs.deployit.plugin.api.reflect.Type;

/**
 * Contains information about the last deployment to an environment or host.
 */
@PublicApiRef
@XmlRootElement(name = "deployment-info")
public class DeploymentInfo extends AbstractDto {

    private static final String FORMAT = "yyyy-MM-dd'T'HH:mm:ss.SSSZ";

    private String id;
    private DateTime lastDeploymentDate;
    private String lastDeploymentBy;
    private Type type;

    public DeploymentInfo() {
    }

    public DeploymentInfo(String id, Type type, DateTime lastDeploymentDate, String lastDeploymentBy) {
        this.id = id;
        this.type = type;
        this.lastDeploymentDate = lastDeploymentDate;
        this.lastDeploymentBy = lastDeploymentBy;
    }

    /**
     * @return Environment or Host id.
     */
    @XmlElement(name = "id")
    public String getId() {
        return id;
    }

    /**
     * @param id environment or host id
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * @return UDM Type of the environment or host
     */
    @XmlElement(name = "type")
    public Type getType() {
        return type;
    }

    public void setType(Type type) {
        this.type = type;
    }

    /**
     * @return Date of last deployment to this environment or host
     */
    @XmlTransient
    public DateTime getLastDeploymentDate() {
        return lastDeploymentDate;
    }

    /**
     * @return Date of last deployment to this environment or host, as string (or <i>null</i>)
     */
    @XmlElement(name = "lastDeploymentDate")
    public String getLastDeploymentDateString() {
        if (lastDeploymentDate == null) {
            return null;
        }
        return DateTimeFormat.forPattern(FORMAT).print(lastDeploymentDate);
    }

    /**
     * @param lastDeploymentDate Date of last deployment to the host or environment
     */
    public void setLastDeploymentDate(DateTime lastDeploymentDate) {
        this.lastDeploymentDate = lastDeploymentDate;
    }

    /**
     * @return User of last deployment on environment.
     */
    @XmlElement(name = "lastDeploymentBy")
    public String getLastDeploymentBy() {
        return lastDeploymentBy;
    }

    /**
     * @param lastDeploymentBy user of last deployment
     */
    public void setLastDeploymentBy(String lastDeploymentBy) {
        this.lastDeploymentBy = lastDeploymentBy;
    }
}
