/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.api;

import com.xebialabs.deployit.engine.api.dto.ArtifactAndData;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemIdAndReferenceId;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemProperties;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xlplatform.documentation.PublicApi;
import org.jboss.resteasy.annotations.providers.multipart.MultipartForm;
import org.joda.time.DateTime;

import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import java.util.List;

/**
 * Provides access to the XL Deploy repository.
 */
@Path("/repository")
@Consumes({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@Produces({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@PublicApi
public interface RepositoryService {

    /**
     * Composes CI prototype filled in with default values.
     *
     * @permission login
     * @param type
     *            the type of the CI
     * @return the CI prototype filled with default values, or a {@code 404} error code if not found.
     */
    @GET
    @Path("ci/new")
    ConfigurationItem construct(@QueryParam("type") Type type);

    /**
     * Reads a configuration item from the repository.
     *
     * @permission read
     * @param ID
     *            the ID of the CI
     * @return the CI, or a {@code 404} error code if not found.
     */
    @GET
    @Path("ci/{ID:.+}")
    ConfigurationItem read(@PathParam("ID") String ID);

    /**
     * Reads a configuration item from the repository.
     *
     * @permission read
     * @param referenceId
     *            the referenceId of the CI
     * @return the CI, or a {@code 404} error code if not found.
     */
    @GET
    @Path("ci/id/{referenceId:.+}")
    ConfigurationItem readByReferenceId(@PathParam("referenceId") String referenceId);

    /**
     * Creates a new configuration item.
     *
     * @permission repo#edit
     * @param ID
     *            the ID/path of the new CI.
     * @param configurationItem
     *            the data of the CI.
     * @return the newly created CI
     */
    @POST
    @Path("ci/{ID:.+}")
    ConfigurationItem create(@PathParam("ID") String ID, ConfigurationItem configurationItem);

    /**
     * Creates a new artifact CI with data.
     *
     * @permission repo#edit
     * @param ID
     *            the ID/path of the new artifact.
     * @param artifactAndData
     *            the data of the artifact.
     * @return the newly created artifact
     */
    @POST
    @Path("ci/{ID:.+}")
    @Consumes({ MediaType.MULTIPART_FORM_DATA })
    ConfigurationItem create(@PathParam("ID") String ID, @MultipartForm ArtifactAndData artifactAndData);

    /**
     * Modifies a configuration item and returns the updated CI if the the update was successful
     * @restDetails otherwise returns a {@code 409} error code if the CI has been updated on the server and you are sending modifications on an old version.
     * @apiDetails otherwise throws a {@link java.lang.RuntimeException RuntimeException} if the CI has been updated on the server and you are sending modifications on an old version.
     *
     * @permission repo#edit
     * @param ID
     *            the ID of the CI to modify.
     * @param configurationItem
     *            the new data of the CI.
     * @return the updated CI if update was successful
     */
    @PUT
    @Path("ci/{ID:.+}")
    ConfigurationItem update(@PathParam("ID") String ID, ConfigurationItem configurationItem);

    /**
     * Modifies an artifact (upload new data) and returns the updated artifact if the the update was successful
     * @restDetails otherwise returns a {@code 409} error code if the CI has been updated on the server and you are sending modifications on an old version.
     * @apiDetails otherwise throws a {@link java.lang.RuntimeException RuntimeException} if the CI has been updated on the server and you are sending modifications on an old version.
     *
     * @permission repo#edit
     * @param ID
     *            the ID of the artifact to modify.
     * @param artifactAndData
     *            the new data of the artifact.
     * @return the updated artifact if update was successful
     */
    @PUT
    @Path("ci/{ID:.+}")
    @Consumes({ MediaType.MULTIPART_FORM_DATA })
    ConfigurationItem update(@PathParam("ID") String ID, @MultipartForm ArtifactAndData artifactAndData);

    /**
     * Deletes a configuration item.
     *
     * @permission repo#edit
     * @permission import#remove
     * @param ID
     *            the ID of the CI to delete.
     */
    @DELETE
    @Path("ci/{ID:.+}")
    void delete(@PathParam("ID") String ID);

    /**
     * Deletes multiple configuration items from the repository.
     *
     * @permission repo#edit
     * @permission import#remove
     * @param ids
     *            the ids of the CIs
     */
    @POST
    @Path("cis/delete")
    void deleteList(List<String> ids);

    /**
     * Reads multiple configuration items from the repository.
     * @restDetails Returns a {@code 404} error code if one of them was not found.
     * @apiDetails Throws {@link java.lang.RuntimeException RuntimeException} if one of them was not found.
     *
     * @permission read
     * @param ids
     *            the ids of the CIs
     * @return the CIs
     */
    @POST
    @Path("cis/read")
    List<ConfigurationItem> read(List<String> ids);

    /**
     * Creates multiple configuration items.
     *
     * @permission repo#edit
     * @param configurationItems
     *            the data of the CIs.
     * @return the newly created CIs
     */
    @POST
    @Path("cis")
    List<ConfigurationItem> create(List<ConfigurationItem> configurationItems);

    /**
     * Modifies multiple configuration items.
     * @restDetails Returns {@code 409} error code if one of the CIs has been updated on the server and you are sending modifications on an old version.
     * @apiDetails Throws {@link java.lang.RuntimeException RuntimeException} if one of the CIs has been updated on the server and you are sending modifications on an old version.
     *
     * @permission repo#edit
     * @param configurationItems
     *            the new data of the CIs.
     * @return the updated CIs if the update was successful

     */
    @PUT
    @Path("cis")
    List<ConfigurationItem> update(List<ConfigurationItem> configurationItems);

    /**
     * Checks if a configuration item exists.
     *
     * @param ID
     *            the ID of the CI to check
     * @return {@code true} if the CI exists, {@code false} if not.
     */
    @GET
    @Path("exists/{ID:.+}")
    Boolean exists(@PathParam("ID") String ID);

    /**
     * Moves a configuration item in the repository.
     *
     * @permission repo#edit
     * @param ID
     *            the ID of the CI to move
     * @param newLocation
     *            the new ID of the CI, reflecting the path in the repository.
     * @return the moved CI
     */
    @POST
    @Path("move/{ID:.+}")
    ConfigurationItem move(@PathParam("ID") String ID, @QueryParam("newId") String newLocation);

    /**
     * Copy a configuration item in the repository. The item as well as all its children are copied to the new location.
     * The parent reflected in the new location, must be of the same type as that of the parent of the item being copied.
     *
     * @permission repo#edit
     * @param ID
     *            the ID of the CI to copy
     * @param newLocation
     *            the new ID of the copied CI, reflecting the path in the repository.
     * @return the copied CI
     */
    @POST
    @Path("copy/{ID:.+}")
    ConfigurationItem copy(@PathParam("ID") String ID, @QueryParam("newId") String newLocation);

    /**
     * Changes the name of a configuration item in the repository.
     *
     * @permission repo#edit
     * @param ID
     *            the ID of the CI to rename
     * @param newName
     *            the new name.
     * @return the renamed CI
     */
    @POST
    @Path("rename/{ID:.+}")
    ConfigurationItem rename(@PathParam("ID") String ID, @QueryParam("newName") String newName);

    /**
     * Retrieves configuration items by way of a query. All parameters are
     * optional.
     *
     * @param type
     *            the type of the CI
     * @param parent
     *            the parent ID of the CI. If set, only the direct children of this CI are searched.
     * @param ancestor
     *            the ancestor ID of the CI. If set, only the subtree of this CI is searched.
     * @param namePattern
     *            a search pattern for the name. This is like the SQL "LIKE"
     *            pattern: the character '%' represents any string of zero or
     *            more characters, and the character '_' (underscore) represents
     *            any single character. Any literal use of these two characters
     *            must be escaped with a backslash ('\'). Consequently, any
     *            literal instance of a backslash must also be escaped,
     *            resulting in a double backslash ('\\').
     * @param lastModifiedBefore look for CIs modified before this date.
     * @param lastModifiedAfter look for CIs modified after this date.
     * @param page the desired page, in case of a paged query.
     * @param resultPerPage the page size, or {@code -1} for no paging.
     * @return a list of references to CIs
     */
    @GET
    @Path("query")
    List<ConfigurationItemId> query(
            @QueryParam("type") Type type,
            @QueryParam("parent") String parent,
            @QueryParam("ancestor") String ancestor,
            @QueryParam("namePattern") String namePattern,
            @QueryParam("lastModifiedBefore") DateTime lastModifiedBefore,
            @QueryParam("lastModifiedAfter") DateTime lastModifiedAfter,
            @QueryParam("page") long page,
            @QueryParam("resultsPerPage") long resultPerPage);

    /**
     * Retrieves configuration items by way of a query. All parameters are
     * optional.
     *
     * @param type
     *            the type of the CI
     * @param parent
     *            the parent ID of the CI. If set, only the direct children of this CI are searched.
     * @param ancestor
     *            the ancestor ID of the CI. If set, only the subtree of this CI is searched.
     * @param namePattern
     *            a search pattern for the name. This is like the SQL "LIKE"
     *            pattern: the character '%' represents any string of zero or
     *            more characters, and the character '_' (underscore) represents
     *            any single character. Any literal use of these two characters
     *            must be escaped with a backslash ('\'). Consequently, any
     *            literal instance of a backslash must also be escaped,
     *            resulting in a double backslash ('\\').
     * @param idPattern
     *            a search pattern for the id. This is like the SQL "LIKE"
     *            pattern: the character '%' represents any string of zero or
     *            more characters, and the character '_' (underscore) represents
     *            any single character. Any literal use of these two characters
     *            must be escaped with a backslash ('\'). Consequently, any
     *            literal instance of a backslash must also be escaped,
     *            resulting in a double backslash ('\\').
     * @param lastModifiedBefore look for CIs modified before this date.
     * @param lastModifiedAfter look for CIs modified after this date.
     * @param page the desired page, in case of a paged query.
     * @param resultPerPage the page size, or {@code -1} for no paging.
     * @return a list of references to CIs
     */
    @GET
    @Path("v2/query")
    List<ConfigurationItemId> queryV2(
            @QueryParam("type") Type type,
            @QueryParam("parent") String parent,
            @QueryParam("ancestor") String ancestor,
            @QueryParam("namePattern") String namePattern,
            @QueryParam("idPattern") String idPattern,
            @QueryParam("lastModifiedBefore") DateTime lastModifiedBefore,
            @QueryParam("lastModifiedAfter") DateTime lastModifiedAfter,
            @QueryParam("page") long page,
            @QueryParam("resultsPerPage") long resultPerPage);

    @GET
    @Path("v3/query")
    List<ConfigurationItemId> queryV3(
            @QueryParam("type") Type type,
            @QueryParam("parent") String parent,
            @QueryParam("ancestor") String ancestor,
            @QueryParam("namePattern") String namePattern,
            @QueryParam("idPattern") String idPattern,
            @QueryParam("lastModifiedBefore") DateTime lastModifiedBefore,
            @QueryParam("lastModifiedAfter") DateTime lastModifiedAfter,
            @QueryParam("page") long page,
            @QueryParam("resultsPerPage") long resultPerPage,
            @QueryParam("properties") ConfigurationItemProperties properties);

    @GET
    @Path("v3/query-extended")
    List<ConfigurationItemIdAndReferenceId> queryV3Extended(
            @QueryParam("type") Type type,
            @QueryParam("parent") String parent,
            @QueryParam("ancestor") String ancestor,
            @QueryParam("namePattern") String namePattern,
            @QueryParam("idPattern") String idPattern,
            @QueryParam("lastModifiedBefore") DateTime lastModifiedBefore,
            @QueryParam("lastModifiedAfter") DateTime lastModifiedAfter,
            @QueryParam("page") long page,
            @QueryParam("resultsPerPage") long resultPerPage,
            @QueryParam("properties") ConfigurationItemProperties properties);

    /**
     * Find candidate values for a property of a {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem}.
     *
     * @param propertyName the property name of the {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem} for which to find candidate values
     * @param namePattern
     *            a search pattern for the name. This is like the SQL "LIKE"
     *            pattern: the character '%' represents any string of zero or
     *            more characters, and the character '_' (underscore) represents
     *            any single character. Any literal use of these two characters
     *            must be escaped with a backslash ('\'). Consequently, any
     *            literal instance of a backslash must also be escaped,
     *            resulting in a double backslash ('\\').
     * @param idPattern
     *            a search pattern for the id. This is like the SQL "LIKE"
     *            pattern: the character '%' represents any string of zero or
     *            more characters, and the character '_' (underscore) represents
     *            any single character. Any literal use of these two characters
     *            must be escaped with a backslash ('\'). Consequently, any
     *            literal instance of a backslash must also be escaped,
     *            resulting in a double backslash ('\\').     * @param page the desired page, in case of a paged query.
     * @param resultPerPage the page size, or {@code -1} for no paging.
     * @param ci The {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem} for which to find the candidate values
     * @return a list of references to CIs
     */
    @POST
    @Path("candidate-values")
    List<ConfigurationItemId> candidateValues(@QueryParam("propertyName") String propertyName,
                                              @QueryParam("namePattern") String namePattern,
                                              @QueryParam("idPattern") String idPattern,
                                              @QueryParam("page") long page,
                                              @QueryParam("resultsPerPage") long resultPerPage,
                                              ConfigurationItem ci);

    /**
     * Validate the configuration items, returning any validation errors found.
     *
     * @param cis The list of configuration items to valdiate.
     * @return The validated configuration items, including any validation messages.
     */
    @POST
    @Path("validate")
    List<ConfigurationItem> validate(@LimitCiTreeDepth(0) List<ConfigurationItem> cis);

    /**
     * Checks if CI is a secure CI
     *
     * @param id id of the CI to check
     *
     * @return {@code true} if the CI is a secure CI, {@code false} if not.
     */
    @GET
    @Path("is-secure/{ID:.+}")
    Boolean isSecure(@PathParam("ID") String id);
}
