/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.api;

import com.xebialabs.deployit.engine.api.dto.XLDAsCodeResult;
import com.xebialabs.xlplatform.documentation.PublicApi;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import java.io.InputStream;

/**
 * XLDAsCode service interface matching XLDAsCodeResource functionality.
 * Provides REST endpoints for DevOps-as-Code operations.
 * 
 * Similar to TaskBlockService, this interface can be accessed by:
 * - In-process workers: ServiceHolder.getXLDAsCodeService()  
 * - External workers: via HTTP proxy from Proxies.getXLDAsCodeService()
 */
@Path("/xld-as-code/v1")
@Consumes({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@Produces({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@PublicApi
public interface XLDAsCodeService {

    /**
     * Applies DevOps-as-Code YAML definition to create/update CIs.
     * Equivalent to XLDAsCodeResource.interpret()
     * 
     * @param yamlContent The YAML definition content
     * @return XLDAsCodeResult indicating success/failure and details
     */
    @POST
    @Path("/apply")
    @Consumes(MediaType.TEXT_PLAIN)
    @Produces(MediaType.APPLICATION_XML)
    XLDAsCodeResult apply(String yamlContent);

    /**
     * Generates YAML definitions from existing XL Deploy repository structure.
     * Equivalent to XLDAsCodeResource.generate()
     * 
     * @param path Repository path to generate from (optional)
     * @param globalPermissions Include global permissions in generation
     * @param roles Include roles in generation  
     * @param users Include users in generation
     * @param includeSecrets Include secrets in generation
     * @param includeDefaults Include default values in generation
     * @return XLDAsCodeResult containing generated YAML content
     */
    @GET
    @Path("/generate")
    @Produces(MediaType.APPLICATION_XML)
    XLDAsCodeResult generate(
        @QueryParam("path") String path,
        @DefaultValue("false") @QueryParam("globalPermissions") boolean globalPermissions,
        @DefaultValue("false") @QueryParam("roles") boolean roles,
        @DefaultValue("false") @QueryParam("users") boolean users,
        @DefaultValue("false") @QueryParam("secrets") boolean includeSecrets,
        @DefaultValue("false") @QueryParam("defaults") boolean includeDefaults
    );
}