package com.xebialabs.xltype.serialization;

import com.xebialabs.deployit.booter.local.LocalBooter;
import com.xebialabs.deployit.engine.api.dto.ValidatedConfigurationItem;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.lookup.LookupValueKey;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import java.net.URL;
import java.util.*;
import java.util.function.Function;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.*;
import static org.testng.Assert.assertNotNull;

@SuppressWarnings("deprecation")
public abstract class CiReaderTestBase<Reader extends CiReader, ListReader extends CiListReader> {
    private String format;
    private ConfigurationItemConverter converter;
    private Function<URL, Reader> constructCiReader;
    private Function<URL, ListReader> constructCiListReader;

    public CiReaderTestBase(String format,
                            Function<URL, Reader> constructCiReader,
                            Function<URL, ListReader> constructCiListReader) {
        this.format = format;
        this.constructCiReader = constructCiReader;
        this.constructCiListReader = constructCiListReader;
    }

    @BeforeClass
    public static void boot() {
        LocalBooter.bootWithoutGlobalContext();
    }

    @BeforeMethod
    public void createConverter() {
        this.converter = new ConfigurationItemConverter();
    }

    //
    // Helper methods
    //

    private URL getTestingResource(String resource) {
        String classpathResource = String.format("%s/%s.%s", format, resource, format);
        return Thread.currentThread().getContextClassLoader().getResource(classpathResource);
    }

    private CiReader createReader(String file) {
        return constructCiReader.apply(getTestingResource(file));
    }

    private CiListReader createListReader(String file) {
        return constructCiListReader.apply(getTestingResource(file));
    }

    private static void assertReference(List<CiReference> references, String ci, String property, String ref) {
        for (CiReference reference : references) {
            if (reference.getCi().getId().equals(ci)
                    && reference.getProperty().getName().equals(property)
                    && reference.getIds().contains(ref)) {
                return;
            }
        }
        assertThat(String.format("Reference '%s=>%s = [%s]' not found in %s", ci, property, ref, references), false);
    }

    //
    // Tests
    //

    @Test
    public void readCi() {
        // Given
        CiReader reader = createReader("ci");

        // When
        ConfigurationItem ci = converter.readCi(reader);

        // Then
        assertThat(ci, notNullValue());
        assertThat(ci.getType(), is(Type.valueOf(CiWithAll.class)));
        assertThat(ci.getId(), is("01"));
        assertThat(ci.getProperty("integer"), is(0));
        assertThat(ci.getProperty("bool"), is(false));
        assertThat(((BaseConfigurationItem) ci).get$token(), is("123"));
    }

    @Test
    public void readCiReference() {
        CiReader reader = createReader("ci-with-reference");

        ConfigurationItem ci = converter.readCi(reader);

        assertThat(ci, notNullValue());
        assertThat(converter.getReferences(), hasSize(1));
        assertReference(converter.getReferences(), "01", "ci", "02");
    }

    @Test
    public void readSetAndListOfCis() {
        CiReader reader = createReader("ci-with-references");

        ConfigurationItem ci = converter.readCi(reader);

        assertThat(ci, notNullValue());
        assertThat(converter.getReferences(), hasSize(2));
        assertReference(converter.getReferences(), "01", "ciSet", "02");
        assertReference(converter.getReferences(), "01", "ciList", "03");
    }

    @Test @SuppressWarnings("unchecked")
    public void readSetAndListOfStrings() {
        CiReader reader = createReader("ci-with-strings");

        ConfigurationItem ci = converter.readCi(reader);

        assertThat(ci.getProperty("stringSet"), equalTo((Set<String>) new HashSet<>(Arrays.asList("a", "b", "c"))));
        assertThat(ci.getProperty("stringList"), equalTo(Arrays.asList("1", "2", "3")));
    }

    @Test @SuppressWarnings("unchecked")
    public void readMapOfStrings() {
        CiReader reader = createReader("ci-with-string-map");

        ConfigurationItem ci = converter.readCi(reader);

        Map<String, String> expected = new HashMap<>();
        expected.put("1", "a");
        expected.put("2", "b");
        assertThat(ci.getProperty("map"), equalTo(expected));
    }

    @Test
    public void readDate() {
        CiReader reader = createReader("ci-with-date");

        ConfigurationItem ci = converter.readCi(reader);

        Calendar calendar = Calendar.getInstance();
        Date date = ci.getProperty("date");
        calendar.setTime(date);
        assertThat(calendar.get(Calendar.YEAR), is(2000));
        assertThat(calendar.get(Calendar.MONTH), is(0));
        assertThat(calendar.get(Calendar.DAY_OF_MONTH), is(1));
    }

    @Test
    public void readEmptyDate() {
        CiReader reader = createReader("ci-with-empty-date");

        ConfigurationItem ci = converter.readCi(reader);

        Date date = ci.getProperty("date");
        assertThat(date, nullValue());
    }

    @Test
    public void readNoValidationMessages() {
        CiReader reader = createReader("ci-with-validation-messages");
        converter.setReadValidationMessages(false);

        ConfigurationItem ci = converter.readCi(reader);

        assertThat(ci, not(instanceOf(ValidatedConfigurationItem.class)));
    }

    @Test
    public void readValidationMessages() {
        CiReader reader = createReader("ci-with-validation-messages");
        converter.setReadValidationMessages(true);

        ConfigurationItem ci = converter.readCi(reader);

        assertThat(ci, instanceOf(ValidatedConfigurationItem.class));

        ValidatedConfigurationItem validatedCi = (ValidatedConfigurationItem) ci;
        assertThat(validatedCi.getValidations(), hasSize(2));

        ValidationMessage message = validatedCi.getValidations().get(0);
        assertThat(message.getCiId(), is("01"));
        assertThat(message.getPropertyName(), is("string"));
        assertThat(message.getMessage(), is("String should not be empty"));
        assertThat(message.getLevel(), is(ValidationMessage.Level.ERROR));
        message = validatedCi.getValidations().get(1);
        assertThat(message.getCiId(), is("01"));
        assertThat(message.getPropertyName(), is("string"));
        assertThat(message.getMessage(), is("This is a warning"));
        assertThat(message.getLevel(), is(ValidationMessage.Level.WARNING));
    }

    @Test
    public void readCis() {
        CiListReader reader = createListReader("list-of-cis");

        List<ConfigurationItem> configurationItems = converter.readCis(reader);

        assertThat(configurationItems.size(), is(2));
        assertThat(configurationItems.get(0).getId(), is("01"));
        assertThat(configurationItems.get(0).getType().toString(), is("test.CiWithAll"));

        assertThat(configurationItems.get(1).getProperty("bool"), is(true));
        assertThat(configurationItems.get(1).getProperty("integer"), is(1));

        assertThat(converter.getReferences(), hasSize(2));
        assertThat(converter.getReferences().get(0).getIds(), hasSize(1));
        assertReference(converter.getReferences(), "01", "ci", "02");
        assertThat(converter.getReferences().get(1).getIds(), hasSize(2));
        assertReference(converter.getReferences(), "02", "ciSet", "01");
        assertReference(converter.getReferences(), "02", "ciSet", "03");
    }

    @Test
    public void readSingleChildAsContainment() {
        CiReader reader = createReader("ci-with-child-as-containment");

        CiWithAll ci = (CiWithAll) converter.readCi(reader);

        assertThat(ci, notNullValue());
        assertThat(ci.getId(), is("01"));
        SonOfCi child = ci.getSingleChild();
        assertNotNull(child);
        assertThat(child.getId(), is("02"));
        assertThat(child.getProperty("string"), is("nested"));
    }

    @Test
    public void readNestedChild() {
        CiReader reader = createReader("ci-with-nested-child");

        CiWithAll ci = (CiWithAll) converter.readCi(reader);

        assertThat(ci, notNullValue());
        assertThat(ci.getId(), is("01"));
        NestedCi child = ci.getNestedChild();
        assertNotNull(child);
        assertThat(child.getId(), is("02"));
        assertThat(child.getProperty("string"), is("nested"));
    }

    @Test
    public void readChildList() {
        CiReader reader = createReader("ci-with-child-list");

        CiWithAll ci = (CiWithAll) converter.readCi(reader);

        assertThat(ci, notNullValue());
        assertThat(ci.getId(), is("01"));
        List<CiWithAll> nestedCIs = ci.getChildList();
        assertThat(nestedCIs, hasSize(2));
        assertThat(nestedCIs.get(0).getId(), is("02"));
        assertThat(nestedCIs.get(0).getProperty("string"), is("nested"));
        assertThat(nestedCIs.get(1).getId(), is("03"));
        assertThat(nestedCIs.get(1).getProperty("string"), is("sibling"));
    }

    @Test
    public void readChildSet() {
        CiReader reader = createReader("ci-with-child-set");

        CiWithAll ci = (CiWithAll) converter.readCi(reader);

        assertThat(ci, notNullValue());
        assertThat(ci.getId(), is("01"));
        Set<CiWithAll> nestedCIs = ci.getChildSet();
        assertThat(nestedCIs, hasSize(2));
    }

    @Test
    public void readListOfNestedCisWithReferences() {
        CiListReader reader = createListReader("list-of-nested-cis-with-references");

        List<ConfigurationItem> cis = converter.readCis(reader);

        assertThat(cis.size(), is(2));
        assertThat(cis.get(0).getId(), is("01"));
        assertThat(cis.get(1).getId(), is("03"));
        CiWithAll ci2 = ((CiWithAll) cis.get(0)).getCi();
        assertThat(ci2.getId(), is("02"));
        assertThat(converter.getReferences(), hasSize(1));
        assertReference(converter.getReferences(), "02", "ci", "03");
    }

    @Test
    public void readCiWithExternalPropertiesLookupValues() {
        CiReader reader = createReader("ci-with-external-properties-lookup-values");
        CiWithAll ci = (CiWithAll)converter.readCi(reader);

        assertThat(ci.get$externalProperties().size(), is(1));

        LookupValueKey aString = (LookupValueKey) ci.get$externalProperties().get("string");

        assertThat(aString.getProviderId(), is("Configurations/Some_Lookup"));
        assertThat(aString.getKey(), is("SOME_KEY"));
    }

    @Test
    public void readCiWithSCMTraceabilityId() {
        CiReader reader = createReader("ci-with-scm-traceability-id");

        BaseConfigurationItem ci = (BaseConfigurationItem) converter.readCi(reader);
        assertThat(ci.get$ciAttributes().getScmTraceabilityDataId(), is(1));
    }
}
