package com.xebialabs.xltype.serialization.json;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.codehaus.jettison.json.JSONArray;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;

import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.xltype.serialization.CiListReader;
import com.xebialabs.xltype.serialization.CiReader;
import com.xebialabs.xltype.serialization.SerializationException;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.newArrayListWithCapacity;
import static com.google.common.collect.Maps.newLinkedHashMap;
import static com.google.common.collect.Sets.newHashSet;

public class CiJsonReader implements CiReader {

    private static final Set<String> SPECIAL_KEYS = newHashSet("id", "type", "$token");

    private final JSONObject json;

    private Iterator<String> propertyIterator;
    private String currentPropertyName;

    public CiJsonReader(JSONObject json) {
        this.json = json;
        this.propertyIterator = properties(json).iterator();
    }

    public static CiJsonReader create(String jsonObject) {
        try {
            return new CiJsonReader(new JSONObject(jsonObject));
        } catch (JSONException e) {
            throw new IllegalArgumentException("Can't parse the following as a JSON object:\n" + jsonObject, e);
        }
    }

    //
    // CiReader implementation
    //

    @Override
    public String getType() {
        try {
            return json.getString("type");
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public String getId() {
        try {
            return json.getString("id");
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public String getToken() {
        try {
            if (json.has("$token")) {
                return json.getString("$token");
            }
            return null;
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public boolean hasMoreProperties() {
        return propertyIterator.hasNext();
    }

    @Override
    public void moveIntoProperty() {
        currentPropertyName = propertyIterator.next();
    }

    @Override
    public CiReader moveIntoNestedProperty() {
        try {
            return new CiJsonReader(json.getJSONObject(currentPropertyName));
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public void moveOutOfProperty() {
        // Noop
    }

    @Override
    public String getCurrentPropertyName() {
        return currentPropertyName;
    }

    @Override
    public String getStringValue() {
        try {
            return json.getString(currentPropertyName);
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public List<String> getStringValues() {
        try {
            JSONArray strings = json.getJSONArray(currentPropertyName);
            return toList(strings);
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public Map<String, String> getStringMap() {
        try {
            JSONObject map = json.getJSONObject(currentPropertyName);
            return toMap(map);
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public boolean isCiReference() {
        return currentPropertyName != null && json.optJSONObject(currentPropertyName) == null;
    }

    @Override
    public String getCiReference() {
        return getStringValue();
    }

    @Override
    public List<String> getCiReferences() {
        return getStringValues();
    }

    @Override
    public CiListReader getCurrentCiListReader() {
        try {
            return new CiListJsonReader(json.getJSONArray(currentPropertyName));
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public List<ValidationMessage> getValidationMessages() {
        List<ValidationMessage> messages = newArrayList();
        try {
            JSONArray array = json.getJSONArray("validation-messages");
            for (int i = 0; i < array.length(); i++) {
                messages.add(toMessage(array.getJSONObject(i)));
            }
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
        return messages;
    }

    //
    // Util
    //

    private static Collection<String> properties(JSONObject object) {
        List<String> properties = newArrayList();
        Iterator<?> keys = object.keys();
        while (keys.hasNext()) {
            String key = keys.next().toString();
            if (SPECIAL_KEYS.contains(key)) {
                continue;
            }
            properties.add(key);
        }

        return properties;
    }

    private static List<String> toList(JSONArray array) throws JSONException {
        List<String> strings = newArrayListWithCapacity(array.length());
        for (int i = 0; i < array.length(); i++) {
            strings.add(array.getString(i));
        }

        return strings;
    }

    private static Map<String, String> toMap(JSONObject object) throws JSONException {
        Map<String, String> map = newLinkedHashMap();
        Iterator<?> keys = object.keys();
        while (keys.hasNext()) {
            String key = keys.next().toString();
            map.put(key, object.getString(key));
        }

        return map;
    }

    private static ValidationMessage toMessage(JSONObject jsonObject) throws JSONException {
        return new ValidationMessage(
            jsonObject.getString("ci"),
            jsonObject.getString("property"),
            jsonObject.getString("message"));
    }

}
