package com.xebialabs.xltype.serialization.rest;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Collection;

import javax.ws.rs.Consumes;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.ext.MessageBodyReader;
import javax.ws.rs.ext.MessageBodyWriter;
import javax.ws.rs.ext.Provider;

import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xltype.serialization.CiListReader;
import com.xebialabs.xltype.serialization.CiReader;
import com.xebialabs.xltype.serialization.CiWriter;
import com.xebialabs.xltype.serialization.ConfigurationItemConverter;

@Provider
@Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
@Consumes({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
public class CiReaderWriter implements MessageBodyWriter<Object>, MessageBodyReader<Object> {

    private Repository repository;
    protected FormatFactory formats = new FormatFactory();

    public CiReaderWriter(Repository repository) {
        this.repository = repository;
    }

    protected ConfigurationItemConverter createConverter() {
        return new ConfigurationItemConverter();
    }

    //
    // Writer
    //

    @Override
    public boolean isWriteable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return isReadable(type, genericType, annotations, mediaType);
    }

    @Override
    public long getSize(Object t, Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return -1L;
    }

    @Override
    @SuppressWarnings("unchecked")
    public void writeTo(Object ci, Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, Object> httpHeaders,
        OutputStream entityStream) throws IOException, WebApplicationException {

        int level = getCiRefsFromLevel();

        ConfigurationItemConverter converter = createConverter();
        CiWriter writer = formats.createCiWriter(mediaType);
        if (Collection.class.isAssignableFrom(type)) {
            converter.writeCis((Collection<ConfigurationItem>) ci, writer, level);
        } else if (ConfigurationItem.class.isAssignableFrom(type)) {
            converter.writeCi((ConfigurationItem) ci, writer, level);
        } else {
            throw new IllegalArgumentException("Can't write objects of type " + type.getName());
        }

        entityStream.write(writer.toString().getBytes());
    }

    /**
     * Returns the maximum depth that child CIs are rendered. Default is {@code 0}, meaning that only simple properties
     * of the top-level CI are written and all child CIs are written as references.
     *
     * @return an integer between {@code 0} and {@link Integer#MAX_VALUE}.
     */
    protected int getCiRefsFromLevel() {
        return 0;
    }

    //
    // Reader
    //

    @Override
    public boolean isReadable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        if (Collection.class.isAssignableFrom(type) && genericType instanceof ParameterizedType) {
            type = (Class<?>) ((ParameterizedType) genericType).getActualTypeArguments()[0];
        }

        return ConfigurationItem.class.isAssignableFrom(type);
    }

    @Override
    public Object readFrom(Class<Object> type, Type genericType, Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, String> httpHeaders,
        InputStream entityStream) throws IOException, WebApplicationException {

        ConfigurationItemConverter converter = createConverter();

        // Parse CIs -- result is either ConfigurationItem or List<ConfigurationItem>
        Object result = null;
        if (Collection.class.isAssignableFrom(type)) {
            CiListReader reader = formats.createCiListReader(entityStream, mediaType);
            result = converter.readCis(reader);
        } else {
            CiReader reader = formats.createCiReader(entityStream, mediaType);
            result = converter.readCi(reader);
        }

        // Resolve the CI references that were found by the converter while parsing
        converter.resolveReferences(repository);

        return result;

    }

}
