/**
 * Copyright © 2014-2016 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xltype.serialization.jdom;

import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.xltype.serialization.CiWriter;
import org.joda.time.DateTime;

import java.util.*;

public class CiJdomWriter implements CiWriter {

    private final JdomWriter jdomWriter;
    // The <list> element is only wrapped around the outer list of CIs. For properties that are LIST_OF_CI or SET_OF_CI,
    // we don't want to have a spurious <list> element. So somehow we need to maintain state whether to render this
    // <list> element is for. This is where the 'suppressListElement' and the corresponding stack come in.
    private Deque<Boolean> suppressListStack = new ArrayDeque<>();

    public CiJdomWriter() {
        this(new JdomWriter());
    }

    public CiJdomWriter(JdomWriter jdomWriter) {
        this.jdomWriter = jdomWriter;
        suppressListStack.push(false);
    }

    @Override
    public void startList() {
        if (suppressListStack.peek()) return;

        jdomWriter.element("list");
    }

    @Override
    public void endList() {
        if (suppressListStack.peek()) return;
        jdomWriter.endElement();
    }

    @Override
    public void startCi(String type, String id) {
        jdomWriter.element(type).attribute("id", id);
    }

    @Override
    public void endCi() {
        jdomWriter.endElement();
    }

    @Override
    public void token(String token) {
        jdomWriter.attribute("token", token);
    }

    @Override
    public void ciAttributes(CiAttributes ciAttributes) {
        addStringAttribute("created-by", ciAttributes.getCreatedBy());
        addDateAttribute("created-at", ciAttributes.getCreatedAt());
        addStringAttribute("last-modified-by", ciAttributes.getLastModifiedBy());
        addDateAttribute("last-modified-at", ciAttributes.getLastModifiedAt());
    }

    @Override
    public void ciFileAttribute(String file) {
        addStringAttribute("file", file);
    }

    private void addStringAttribute(String attrName, String attrValue) {
        if (attrValue != null) {
            jdomWriter.attribute(attrName, attrValue);
        }
    }

    private void addDateAttribute(String attrName, DateTime attrValue) {
        if (attrValue != null) {
            jdomWriter.attributeAsDate(attrName, attrValue);
        }
    }

    @Override
    public void startProperty(String name) {
        jdomWriter.element(name);
        suppressListStack.push(true);
    }

    @Override
    public void endProperty() {
        jdomWriter.endElement();
        suppressListStack.pop();
    }

    @Override
    public void valueAsString(Object value) {
        jdomWriter.value(value.toString());
    }

    @Override
    public void valuesAsStrings(Collection<?> values) {
        for (Object value : values) {
            jdomWriter.element("value").value(value.toString()).endElement();
        }
    }

    @Override
    public void mapAsStrings(Map<?, ?> map) {
        for (Map.Entry<?, ?> entry : map.entrySet()) {
            jdomWriter.element("entry").attribute("key", entry.getKey().toString()).value(entry.getValue().toString()).endElement();
        }
    }

    @Override
    public void ciReference(String reference) {
        jdomWriter.attribute("ref", reference);
    }

    @Override
    public void ciReferences(Collection<String> references) {
        for (String reference : references) {
            jdomWriter.element("ci").attribute("ref", reference).endElement();
        }
    }

    @Override
    public void typedCiReference(ConfigurationItemId ci) {
        jdomWriter.element("ci");
        jdomWriter.attribute("ref", ci.getId());
        if (ci.getType() != null) {
            jdomWriter.attribute("type", ci.getType().toString());
        }
        jdomWriter.endElement();
    }

    @Override
    public void typedCiReferences(Collection<ConfigurationItemId> references) {
        jdomWriter.element("list");
        for (ConfigurationItemId reference : references) {
            typedCiReference(reference);
        }
        jdomWriter.endElement();
    }

    @Override
    public void validationMessages(List<ValidationMessage> validations) {
        jdomWriter.element("validation-messages");
        for (ValidationMessage validation : validations) {
            jdomWriter.element("validation-message");
            jdomWriter.attribute("level", validation.getLevel().name());
            jdomWriter.attribute("ci", validation.getCiId());
            if (validation.getPropertyName() != null) {
                jdomWriter.attribute("property", validation.getPropertyName());
            }
            jdomWriter.value(validation.getMessage());
            jdomWriter.endElement();
        }
        jdomWriter.endElement();
    }

    @Override
    public String toString() {
        return jdomWriter.toString();
    }
}
