/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xltype.serialization.json;

import com.xebialabs.deployit.engine.api.dto.DeploymentInfo;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import com.xebialabs.deployit.plugin.api.reflect.Type;

import static com.xebialabs.deployit.checks.Checks.checkArgument;

public class DeploymentInfoJsonConverter {

    private DateTimeFormatter FORMATTER = DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss.SSSZ");

    public DeploymentInfoJsonConverter() {
    }

    public String toJson(DeploymentInfo deploymentInfo) {
        JsonWriter writer = new JsonWriter();
        toJson(deploymentInfo, writer);
        return writer.toString();
    }

    public void toJson(final DeploymentInfo deploymentInfo, final JsonWriter writer) {
        writer.object();

        writer.key("id").value(deploymentInfo.getId());
        DateTime lastDeploymentDate = deploymentInfo.getLastDeploymentDate();
        writer.key("lastDeploymentDate")
                .value(lastDeploymentDate == null ? null : FORMATTER.print(deploymentInfo.getLastDeploymentDate()));
        writer.key("lastDeploymentBy").value(deploymentInfo.getLastDeploymentBy());
        writer.key("type").value(deploymentInfo.getType());

        writer.endObject();
    }

    public DeploymentInfo toDeploymentInfo(String jsonString) {
        try {
            JSONObject json = new JSONObject(jsonString);

            checkArgument(json.has("id"), "Missing 'id' property in JSON:\n%s", jsonString);
            checkArgument(json.has("lastDeploymentDate"), "Missing 'lastDeploymentDate' property in JSON:\n%s", jsonString);
            checkArgument(json.has("lastDeploymentBy"), "Missing 'lastDeploymentBy' property in JSON:\n%s", jsonString);
            checkArgument(json.has("type"), "Missing 'type' property in JSON:\n%s", jsonString);
            return new DeploymentInfo(
                    json.getString("id"),
                    Type.valueOf(json.getString("type")),
                    DateTime.parse(json.getString("lastDeploymentDate"), FORMATTER),
                    json.getString("lastDeploymentBy")
            );

        } catch (JSONException e) {
            throw new IllegalArgumentException("Can't parse JSON:\n" + jsonString, e);
        }
    }
}
