/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xltype.serialization.json;

import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;

import com.xebialabs.deployit.engine.api.dto.Control;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Parameters;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItemWithPolicies;
import com.xebialabs.xltype.serialization.CiWriter;
import com.xebialabs.xltype.serialization.ConfigurationItemConverter;

import static com.xebialabs.xltype.serialization.json.Converters.checkArgument;

public class ControlJsonConverter {

    private final ConfigurationItemConverter ciConverter;

    public ControlJsonConverter(ConfigurationItemConverter converter) {
        this.ciConverter = converter;
    }

    public ConfigurationItemConverter getCiConverter() {
        return ciConverter;
    }

    public String toJson(Control control) {
        JsonWriter writer = new JsonWriter();
        CiWriter ciWriter = new CiJsonWriter(writer);

        writer.object();
        writer.key("method").value(control.getControlName());

        ConfigurationItem ci = control.getConfigurationItem();
        writer.key("controllable");
        ciConverter.writeCi(ci, ciWriter);

        addPolicies(ci, writer);

        if (control.getParameters() != null) {
            writer.key("parameters");
            ciConverter.writeCi(control.getParameters(), ciWriter);
        }

        writer.endObject();

        return writer.toString();
    }

    public Control toControl(String jsonString) {
        try {
            JSONObject json = new JSONObject(jsonString);

            checkArgument(json.has("method"), "Missing 'method' property in JSON:\n%s", jsonString);
            checkArgument(json.has("controllable"), "Missing 'controllable' property in JSON:\n%s", jsonString);

            String name = json.getString("method");
            ConfigurationItem ci = ciConverter.readCi(new CiJsonReader(json.getJSONObject("controllable")));
            Parameters parameters = getParameters(json.optJSONObject("parameters"));

            return new Control(ci, name, parameters);

        } catch (JSONException e) {
            throw new IllegalArgumentException("Can't parse JSON:\n" + jsonString, e);
        }
    }

    private void addPolicies(ConfigurationItem ci, JsonWriter writer) {
        if (ci instanceof BaseConfigurationItemWithPolicies) {
            BaseConfigurationItemWithPolicies ciWithPolicies = (BaseConfigurationItemWithPolicies) ci;
            writer.key("policies").object();
            writer.key("onSuccessPolicy").value(ciWithPolicies.getOnSuccessPolicy());
            writer.key("onFailurePolicy").value(ciWithPolicies.getOnFailurePolicy());
            writer.endObject();
        }
    }

    private Parameters getParameters(final JSONObject jsonObject) {
        return jsonObject == null ? null : (Parameters) ciConverter.readCi(new CiJsonReader(jsonObject));
    }
}
