/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xltype.serialization.xstream;

import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import com.xebialabs.deployit.engine.api.execution.*;
import com.xebialabs.deployit.plugin.api.deployment.ResolvedPlaceholder;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import java.util.*;

public class Converters {

    public static <T> List<T> readList(Object owner, Class<T> toRead, HierarchicalStreamReader reader, UnmarshallingContext context) {
        List<T> list = new ArrayList<>();
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            @SuppressWarnings("unchecked")
            T child = (T) context.convertAnother(owner, toRead);
            list.add(child);
            reader.moveUp();
        }
        return list;
    }

    public static Set<ResolvedPlaceholder> readPlaceholders(HierarchicalStreamReader reader) {
        Set<ResolvedPlaceholder> placeholders = new HashSet<>();
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            boolean encrypted = Boolean.parseBoolean(reader.getAttribute("encrypted"));
            String containerId = reader.getAttribute("containerId");
            boolean containerDeleted = Boolean.parseBoolean(reader.getAttribute("containerDeleted"));
            String dictionaryId = reader.getAttribute("dictionaryId");
            boolean dictionaryDeleted = Boolean.parseBoolean(reader.getAttribute("dictionaryDeleted"));
            String deployedAppId = reader.getAttribute("deployedAppId");
            String environmentId = reader.getAttribute("environmentId");
            boolean environmentDeleted = Boolean.parseBoolean(reader.getAttribute("environmentDeleted"));
            String versionId = reader.getAttribute("versionId");
            String key = reader.getAttribute("key");
            String value = reader.getAttribute("value");
            placeholders.add(new ResolvedPlaceholder(encrypted, key, value, containerId, containerDeleted, deployedAppId,
                    dictionaryId, dictionaryDeleted, environmentId, environmentDeleted, versionId));
            reader.moveUp();
        }
        return placeholders;
    }

    public static Map<String, String> readMap(HierarchicalStreamReader reader) {
        Map<String, String> map = new HashMap<>();
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            map.put(reader.getAttribute("key"), reader.getValue());
            reader.moveUp();
        }
        return map;
    }

    public static void writeNode(String name, String value, HierarchicalStreamWriter writer) {
        writer.startNode(name);
        if (value != null) {
            writer.setValue(value);
        }
        writer.endNode();
    }

    public static void writeConfigurationItem(ConfigurationItem ci, HierarchicalStreamWriter writer, MarshallingContext context) {
        writer.startNode(ci.getType().toString());
        context.convertAnother(ci);
        writer.endNode();
    }

    public static void writePhaseContainer(PhaseContainerState phaseContainerState, HierarchicalStreamWriter writer, MarshallingContext context, boolean writeStepBlocks) {

        writeBlockBasics(phaseContainerState, writer);
        writer.addAttribute("root", "true");

        for (PhaseState phase : phaseContainerState.getBlocks()) {
            writePhase(phase, writer, context, writeStepBlocks);
        }
    }

    public static void writePhase(PhaseState phase, HierarchicalStreamWriter writer, MarshallingContext context, boolean writeStepBlocks) {

        writer.startNode("block");
        writeBlockBasics(phase, writer);
        writer.addAttribute("phase", "true");

        writer.startNode("block");
        writeBlockTree(phase.getBlock(), writer, context, writeStepBlocks);
        writer.endNode();

        writer.endNode();
    }

    public static void writeBlockTree(BlockState block, HierarchicalStreamWriter writer, MarshallingContext context, boolean writeStepBlocks) {

        if (block instanceof CompositeBlockState) {
            writeBlockBasics(block, writer);
            CompositeBlockState cbs = (CompositeBlockState) block;
            writer.addAttribute("parallel", String.valueOf(cbs.isParallel()));
            for (BlockState blockState : cbs.getBlocks()) {
                writer.startNode("block");
                writeBlockTree(blockState, writer, context, writeStepBlocks);
                writer.endNode();
            }
        } else if (writeStepBlocks && block instanceof StepBlockState) {
            writeStepBlock((StepBlockState) block, writer, context);
        } else if (block instanceof StepBlockState) {
            writeBlockBasics(block, writer);
        } else {
            throw new IllegalArgumentException("Can only write Composite or Step Blocks, got: " + block);
        }

    }

    public static void writeStepBlock(StepBlockState block, HierarchicalStreamWriter writer, MarshallingContext context) {
        writeBlockBasics(block, writer);
        writer.addAttribute("current", String.valueOf(block.getCurrentStep()));
        writeSteps(block.getSteps(), writer, context);
    }

    public static void writeSteps(List<StepState> steps, HierarchicalStreamWriter writer, MarshallingContext context) {
        for (StepState step : steps) {
            writer.startNode("step");
            context.convertAnother(step);
            writer.endNode();
        }
    }

    public static void writeBlockBasics(BlockState block, HierarchicalStreamWriter writer) {
        writer.addAttribute("id", block.getId() == null ? "" : block.getId());
        writer.addAttribute("state", block.getState().toString());
        writer.addAttribute("description", block.getDescription());
        writer.addAttribute("hasSteps", block.hasSteps().toString());
        if (block.getSatelliteConnectionState() != null)
            writer.addAttribute("satelliteConnectionState", block.getSatelliteConnectionState().toString());
    }

}
