package com.xebialabs.deployit.engine.xml;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;

import java.util.Collection;
import java.util.Map;

import static com.google.common.base.Strings.nullToEmpty;
import static com.google.common.collect.Maps.newHashMap;

public abstract class AbstractConfigurationItemConverter implements Converter {

    @SuppressWarnings("unchecked")
    @Override
    public void marshal(Object source, HierarchicalStreamWriter writer, MarshallingContext context) {
        ConfigurationItem ci = (ConfigurationItem) source;
        writer.addAttribute("id", ci.getId());
        if (ci instanceof BaseConfigurationItem) {
            writer.addAttribute("token", nullToEmpty(((BaseConfigurationItem) ci).get$token()));
        }
        writeProperties(ci, writer);
    }

    @Override
    public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
        Type type = Type.valueOf(reader.getNodeName());
        String id = reader.getAttribute("id");
        Descriptor descriptor = type.getDescriptor();
        if (descriptor == null) {
            throw new IllegalStateException("Encountered unknown CI type [" + type + "] for ConfigurationItem [" + id + "]");
        }
        ConfigurationItem configurationItem = descriptor.newInstance();
        configurationItem.setId(id);
        if (configurationItem instanceof BaseConfigurationItem) {
            ((BaseConfigurationItem) configurationItem).set$token(reader.getAttribute("token"));
        }
        configurationItem = readProperties(reader, descriptor, configurationItem);
        return configurationItem;
    }

    protected void writeProperties(ConfigurationItem ci, HierarchicalStreamWriter writer) {
        for (PropertyDescriptor propertyDescriptor : ci.getType().getDescriptor().getPropertyDescriptors()) {
            if (!propertyDescriptor.isHidden()) {
                writeProperty(ci, propertyDescriptor, writer);
            }
        }
    }

    protected void writeProperty(ConfigurationItem ci, PropertyDescriptor propertyDescriptor, HierarchicalStreamWriter writer) {
        Object value = propertyDescriptor.get(ci);
        if (value == null) return;
        writer.startNode(propertyDescriptor.getName());
        switch (propertyDescriptor.getKind()) {
            case STRING:
                writeStringProperty(value, propertyDescriptor, writer);
                break;
            case BOOLEAN:
            case INTEGER:
            case ENUM:
                writer.setValue(value.toString());
                break;
            case CI:
                writeCiProperty(value, propertyDescriptor, writer);
                break;
            case SET_OF_STRING:
            case LIST_OF_STRING:
                writeCollectionOfStringProperty(value, propertyDescriptor, writer);
                break;
            case SET_OF_CI:
            case LIST_OF_CI:
                writeCollectionOfCiProperty(value, propertyDescriptor, writer);
                break;
            case MAP_STRING_STRING:
                writeMapStringStringProperty(value, propertyDescriptor, writer);
                break;
        }
        writer.endNode();
    }

    protected void writeStringProperty(Object value, PropertyDescriptor propertyDescriptor, HierarchicalStreamWriter writer) {
        writer.setValue(value.toString());
    }

    protected void writeMapStringStringProperty(Object value, PropertyDescriptor propertyDescriptor, HierarchicalStreamWriter writer) {
        for (Map.Entry<String, String> entry : ((Map<String, String>) value).entrySet()) {
            writer.startNode("entry");
            writer.addAttribute("key", entry.getKey());
            writer.setValue(entry.getValue());
            writer.endNode();
        }
    }

    protected void writeCollectionOfStringProperty(Object value, PropertyDescriptor propertyDescriptor, HierarchicalStreamWriter writer) {
        for (String s : (Collection<String>) value) {
            writer.startNode("value");
            writer.setValue(s);
            writer.endNode();
        }
    }

    protected ConfigurationItem readProperties(HierarchicalStreamReader reader, Descriptor descriptor, ConfigurationItem configurationItem) {
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            readProperty(reader, descriptor, configurationItem);
            reader.moveUp();
        }

        return configurationItem;
    }

    protected void readProperty(HierarchicalStreamReader reader, Descriptor descriptor, ConfigurationItem configurationItem) {
        String propName = reader.getNodeName();
        PropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor(propName);
        if (propertyDescriptor == null) {
            throw new IllegalStateException("Encountered unknown ConfigurationItem property [" + descriptor.getType() + "." + propName + "] for ci [" + configurationItem.getId() + "]");
        }
        switch (propertyDescriptor.getKind()) {
            case STRING:
                readStringProperty(configurationItem, propertyDescriptor, reader);
                break;
            case BOOLEAN:
            case INTEGER:
            case ENUM:
                propertyDescriptor.set(configurationItem, reader.getValue());
                break;
            case CI:
                readCiProperty(configurationItem, propertyDescriptor, reader);
                break;
            case SET_OF_STRING:
            case LIST_OF_STRING:
                readCollectionOfStringProperty(configurationItem, propertyDescriptor, reader);
                break;
            case SET_OF_CI:
            case LIST_OF_CI:
                readCollectionOfCiProperty(configurationItem, propertyDescriptor, reader);
                break;
            case MAP_STRING_STRING:
                readMapStringStringProperty(configurationItem, propertyDescriptor, reader);
                break;
        }
    }

    protected void readStringProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, HierarchicalStreamReader reader) {
        propertyDescriptor.set(configurationItem, reader.getValue());
    }

    protected void readMapStringStringProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, HierarchicalStreamReader reader) {
        Map<String, String> map = newHashMap();
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            map.put(reader.getAttribute("key"), reader.getValue());
            reader.moveUp();
        }
        propertyDescriptor.set(configurationItem, map);
    }

    protected void readCollectionOfStringProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, HierarchicalStreamReader reader) {
        Collection<String> strings = (propertyDescriptor.getKind() == PropertyKind.SET_OF_STRING ? Sets.<String>newHashSet() : Lists.<String>newArrayList());
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            strings.add(reader.getValue());
            reader.moveUp();
        }
        propertyDescriptor.set(configurationItem, strings);
    }

    @Override
    public boolean canConvert(Class type) {
        return ConfigurationItem.class.isAssignableFrom(type);
    }

    protected abstract void writeCiProperty(Object value, PropertyDescriptor propertyDescriptor, HierarchicalStreamWriter writer);

    protected abstract void writeCollectionOfCiProperty(Object value, PropertyDescriptor propertyDescriptor, HierarchicalStreamWriter writer);

    protected abstract void readCiProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, HierarchicalStreamReader reader);

    protected abstract void readCollectionOfCiProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, HierarchicalStreamReader reader);
}
