package com.xebialabs.deployit.booter.local;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;

import com.xebialabs.deployit.booter.local.utils.ReflectionUtils;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.reflect.MethodDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.MethodVerification;
import com.xebialabs.deployit.plugin.api.reflect.VerificationContext;
import com.xebialabs.deployit.plugin.api.reflect.Verify;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.ControlTask;
import com.xebialabs.deployit.plugin.api.udm.Delegate;
import com.xebialabs.deployit.plugin.api.udm.Parameters;

import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.booter.local.LocalMethodDescriptor.CONTROL_TASK_DISPATCH_METHOD;

/**
 * Used through reflection...
 */
@SuppressWarnings("UnusedDeclaration")
public class DefaultDelegates {

    @SuppressWarnings("unchecked")
    @Delegate(name = "methodInvoker")
    public static List<Step> invokeMethod(ConfigurationItem ci, final String methodName, Map<String, String> arguments, Parameters parameters) {
        ArrayList<Method> methods = newArrayList(ci.getClass().getMethods());
        Method method = Iterables.find(methods, new Predicate<Method>() {
            public boolean apply(final Method input) {
                return input.getName().equals(methodName) && input.isAnnotationPresent(ControlTask.class);
            }
        });

        try {
            if (method.getParameterTypes().length == 1) {
                return (List<Step>) method.invoke(ci, parameters);
            } else {
                return (List<Step>) method.invoke(ci);
            }
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Could not invoke " + methodName + " on " + ci, e);
        } catch (InvocationTargetException e) {
            throw ReflectionUtils.handleInvocationTargetException(e, "Could not invoke " + methodName + " on " + ci);
        }
    }

    @SuppressWarnings("unchecked")
    @Delegate(name = "dispatcherInvoker")
    @DispatcherVerification
    public static List<Step> invokeDispatcher(ConfigurationItem ci, final String methodName, Map<String, String> arguments, Parameters parameters) {
        Method dispatcher = getDispatcher(ci);

        try {
            if (dispatcher.getParameterTypes().length == 3) {
                return (List<Step>) dispatcher.invoke(ci, methodName, arguments, parameters);
            } else if (dispatcher.getParameterTypes().length == 2) {
                return (List<Step>) dispatcher.invoke(ci, methodName, arguments);
            }
            return (List<Step>) dispatcher.invoke(ci, methodName);
        } catch (InvocationTargetException e) {
            throw ReflectionUtils.handleInvocationTargetException(e, "Could not invoke " + methodName + " on " + ci);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Could not invoke " + methodName + " on " + ci, e);
        }
    }

    private static Method getDispatcher(final ConfigurationItem ci) {
        Class<?> clazz = ci.getClass();
        return filter(newArrayList(clazz.getMethods()), new Predicate<Method>() {
            @Override
            public boolean apply(final Method input) {
                return input.getName().equals(CONTROL_TASK_DISPATCH_METHOD);
            }
        }).iterator().next();
    }

    @Retention(RetentionPolicy.RUNTIME)
    @Target(ElementType.METHOD)
    @Verify(clazz = DispatcherVerification.Verification.class, type = "dispatcherVerification")
    static @interface DispatcherVerification {
        public static class Verification implements MethodVerification {
            @Override
            public void verify(final MethodDescriptor descriptor, final VerificationContext context) {
                Class<?> clazz = ((LocalMethodDescriptor) descriptor).getDescriptor().getClazz();
                List<Method> filter = newArrayList(filter(newArrayList(clazz.getMethods()), new Predicate<Method>() {
                    public boolean apply(final Method input) {
                        return input.getName().equals(CONTROL_TASK_DISPATCH_METHOD);
                    }
                }));
                if (filter.size() == 0) {
                    context.error("ControlTask dispatcher [%s] for [%s] is not present.", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn());
                } else if (filter.size() > 1) {
                    context.error("Found more than 1 ControlTask dispatcher [%s] for [%s].", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn());
                } else {
                    Method method = filter.get(0);
                    Class<?>[] parameterTypes = method.getParameterTypes();

                    boolean methodWithArgumentsAndParameters = parameterTypes.length == 3;
                    boolean methodWithOnlyArguments = parameterTypes.length == 2;

                    if (!List.class.isAssignableFrom(method.getReturnType())) {
                        context.error("ControlTask dispatcher [%s] for [%s] should return a List<Step>", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn());
                    }

                    if (!String.class.isAssignableFrom(parameterTypes[0])) {
                        context.error("ControlTask dispatcher [%s] for [%s] doesn't take a String as first parameter.", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn());
                    }

                    if ((methodWithArgumentsAndParameters || methodWithOnlyArguments) && !Map.class.isAssignableFrom(parameterTypes[1])) {
                        context.error("ControlTask dispatcher [%s] for [%s] should take a Map<String,String> as second parameter.", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn());
                    }

                    if (methodWithArgumentsAndParameters && !Parameters.class.isAssignableFrom(parameterTypes[2])) {
                        context.error("ControlTask dispatcher [%s] for [%s] should take a [udm.Parameters] sub-type as third parameter.", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn());
                    }

                    if (descriptor.getParameterObjectType() != null && !methodWithArgumentsAndParameters) {
                        context.error("ControlTask dispatcher [%s] for [%s] doesn't take parameters of type [%s].", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn(), descriptor.getParameterObjectType());
                    }

                    if (descriptor.getParameterObjectType() == null && descriptor.getAttributes().size() > 0 && !methodWithOnlyArguments) {
                        context.error("ControlTask dispatcher [%s] for [%s] doesn't take a Map of arguments", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn());
                    }
                }
            }
        }
    }

}
