package com.xebialabs.deployit.booter.local;

import com.xebialabs.deployit.booter.local.validation.BooleanOrPlaceholderValidator;
import com.xebialabs.deployit.booter.local.validation.ConvertingValidationRule;
import com.xebialabs.deployit.booter.local.validation.IntegerOrPlaceholderValidator;
import com.xebialabs.deployit.plugin.api.reflect.InputHint;
import com.xebialabs.deployit.plugin.api.reflect.InputHintValue;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.Validator;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static com.xebialabs.deployit.plugin.api.reflect.InputHintValue.inputHintValue;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.ENUM;
import static java.lang.String.format;

class GeneratedDeployablePropertyDescriptor extends LocalPropertyDescriptor {

    public GeneratedDeployablePropertyDescriptor(LocalDescriptor descriptor, PropertyDescriptor deployedPropertyDescriptor) {
        setName(deployedPropertyDescriptor.getName());
        setDeclaringDescriptor(descriptor);
        initMetadataFromDescriptor(deployedPropertyDescriptor);
    }

    private void initMetadataFromDescriptor(PropertyDescriptor propertyDescriptor) {
        // Generated 'simple' Deployable properties always are of kind String so that placeholder/dictionaries values can be filled.
        setKind(propertyDescriptor.getKind().isSimple() ? PropertyKind.STRING : propertyDescriptor.getKind());
        setCategory(propertyDescriptor.getCategory());
        setLabel(propertyDescriptor.getLabel());
        setRequired(false);
        setPassword(propertyDescriptor.isPassword());
        setSize(propertyDescriptor.getSize());
        initDescription(propertyDescriptor);
        setInputHint(createInputHint(propertyDescriptor));
    }

    private void initDescription(PropertyDescriptor propertyDescriptor) {
        String description = format("%s (%s)", propertyDescriptor.getDescription(), propertyDescriptor.getKind().name().toLowerCase());
        if (propertyDescriptor.getKind() == ENUM) {
            description += format(" values(%s)", String.join(", ", propertyDescriptor.getEnumValues()));
        }
        if (propertyDescriptor.getKind().isSimple() && propertyDescriptor.getDefaultValue() != null) {
            description += format(" default(%s)", propertyDescriptor.getDefaultValue());
        }
        setDescription(description);
    }

    private InputHint createInputHint(PropertyDescriptor propertyDescriptor) {
        return (new InputHint.InputHintBuilder()).construct($ -> {
            $.kind = propertyDescriptor.getKind();
            $.required = propertyDescriptor.isRequired();
            $.values.addAll(getEnumValues(propertyDescriptor));
            $.validationRules = getValidationRules(propertyDescriptor);
            $.prompt = getPrompt(propertyDescriptor.getInputHint());
            $.copyFromProperty = getCopyFromProperty(propertyDescriptor.getInputHint());
        }).constructInputHint();
    }

    private List<InputHintValue> getEnumValues(PropertyDescriptor propertyDescriptor) {
        List<InputHintValue> inputHintValues= new ArrayList<>();
        List<String> enumValues = propertyDescriptor.getEnumValues();
        if (enumValues != null) {
            inputHintValues.addAll(enumValues.stream().map(value -> inputHintValue(value, value)).collect(Collectors.toList()));
        }
        if (propertyDescriptor.getInputHint() != null && propertyDescriptor.getInputHint().getValues() != null)
            inputHintValues.addAll(propertyDescriptor.getInputHint().getValues());
        return inputHintValues;
    }

    private String getPrompt(InputHint inputHint) {
        if (inputHint != null)
            return inputHint.getPrompt();
        return null;
    }

    private String getCopyFromProperty(InputHint inputHint) {
        if (inputHint != null)
            return inputHint.getCopyFromProperty();
        return null;
    }

    private Set<Validator<?>> getValidationRules(PropertyDescriptor propertyDescriptor) {
        Set<Validator<?>> validationRules = new HashSet<>();
        if (propertyDescriptor instanceof LocalPropertyDescriptor) {
            LocalPropertyDescriptor localPropertyDescriptor = (LocalPropertyDescriptor) propertyDescriptor;
            if (localPropertyDescriptor.getKind().isSimple()) {
                for (Validator<?> v : localPropertyDescriptor.validationRules) {
                    validationRules.add(new ConvertingValidationRule<>(v, localPropertyDescriptor.createConverter()));
                }
            } else {
                validationRules.addAll(localPropertyDescriptor.validationRules);
            }
            if (propertyDescriptor.getInputHint() != null && propertyDescriptor.getInputHint().getValidationRules() != null)
                validationRules.addAll(propertyDescriptor.getInputHint().getValidationRules());
            addPropertyKindValiationRule(propertyDescriptor.getKind(), validationRules);
        }
        return validationRules;
    }

    private void addPropertyKindValiationRule(PropertyKind propertyKind, Set<Validator<?>> validators) {
        switch (propertyKind) {
            case BOOLEAN:
                validators.add(new BooleanOrPlaceholderValidator());
                break;
            case INTEGER:
                validators.add(new IntegerOrPlaceholderValidator());
                break;
        }
    }

    @Override
    public Object get(ConfigurationItem item) {
        return getDeclaringDescriptor().getSyntheticPropertyValue(item, getName());
    }

    @Override
    protected void doSetValue(ConfigurationItem item, Object value) {
        getDeclaringDescriptor().setSyntheticPropertyValue(item, getName(), value);
    }
}
