/**
 * Copyright © 2014-2015 XebiaLabs B.V. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.packager;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableList;
import com.google.common.io.CharStreams;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Application;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Version;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import de.schlichtherle.truezip.file.TFile;

import javax.xml.bind.DatatypeConverter;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.*;

import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.packager.CiNameCache.id;
import static com.xebialabs.deployit.packager.truezip.TFiles.newOutputSupplier;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.APPLICATIONS;

public class ManifestWriterDriver {

    public static final HashSet<String> APPLICATION_PROPERTIES_TO_SKIP = newHashSet("application");

    public static final Set<String> ARTIFACT_PROPERTIES_TO_SKIP = newHashSet("placeholders");

    private static enum WritePasswords {
        YES, NO, PLACEHOLDERS_ONLY
    }

    private final CiNameCache cache;
    private ManifestWriter writer;

    public ManifestWriterDriver(CiNameCache cache, ManifestWriter writer) {
        this.writer = writer;
        this.cache = cache;
    }

    public void writeToPackage(Version version, TFile file) throws IOException {
        writeToPackage(version, file, false);
    }

    public void writeToPackage(Version version, TFile file, boolean writeFullAppPath) throws IOException {
        TFile manifestFile = new TFile(file, writer.getManifestFileName());
        write(version, writeFullAppPath);
        CharStreams.write(writer.toString(), CharStreams.newWriterSupplier(newOutputSupplier(manifestFile), Charset.defaultCharset()));
    }

    @VisibleForTesting
    void write(Version version) {
        write(version, false);
    }

    @VisibleForTesting
    void write(Version version, boolean writeFullAppPath) {
        cache.scan(version);
        writeVersion(version, writeFullAppPath);
        writer.endManifest();
    }

    private void writeProperties(ConfigurationItem ci, WritePasswords writePasswords, ManifestWriter.ManifestCiWriter ciWriter) {
        for (PropertyDescriptor propertyDescriptor : ci.getType().getDescriptor().getPropertyDescriptors()) {
            if (!shouldSkipFileUri(propertyDescriptor, ci)) {
                writeProperty(propertyDescriptor, ci, writePasswords, ciWriter);
            }
        }
    }

    private ManifestWriter.ManifestCiWriter writeVersion(Version version, boolean writeFullAppPath) {
        String appName = writeFullAppPath
                ? applicationNameWithDirectoryPathPreserved(version.getApplication())
                : cache.lookup(version.getApplication());
        ManifestWriter.ManifestCiWriter manifestCiWriter = writer.writeVersion(version.getType(), cache.lookup(version), appName);
        WritePasswords exposePasswords = shouldWritePasswords(version);

        for (PropertyDescriptor pd : version.getType().getDescriptor().getPropertyDescriptors()) {
            if (!APPLICATION_PROPERTIES_TO_SKIP.contains(pd.getName())) {
                writeProperty(pd, version, exposePasswords, manifestCiWriter);
            }
        }
        return manifestCiWriter;
    }

    private WritePasswords shouldWritePasswords(Version version) {
        boolean exportAllPasswords = version.getProperty("exportAllPasswords");
        boolean exportOnlyPasswordPlaceholders = version.getProperty("exportOnlyPasswordPlaceholders");
        if (exportAllPasswords) {
            return WritePasswords.YES;
        } else if (exportOnlyPasswordPlaceholders) {
            return WritePasswords.PLACEHOLDERS_ONLY;
        }
        return WritePasswords.NO;
    }

    private String applicationNameWithDirectoryPathPreserved(Application app) {
        if (app.getId().startsWith(APPLICATIONS.getRootNodeName())) {
            return app.getId().substring(APPLICATIONS.getRootNodeName().length() + 1);
        } else {
            return app.getId();
        }
    }

    private String filterPasswordValue(String value, WritePasswords writePasswords) {
        if (writePasswords == WritePasswords.YES) {
            return value;
        } else if (writePasswords == WritePasswords.PLACEHOLDERS_ONLY && value.startsWith("{{") && value.endsWith("}}")) {
            return value;
        }
        return "";
    }

    Type lookupType(ConfigurationItem ci, Class<?> ciClass) {
        if (ci.getType().getTypeSource() != null)
            return DescriptorRegistry.getDescriptorRegistry(ci.getType().getTypeSource()).lookupType(ciClass);
        return Type.valueOf(ciClass);
    }

    @SuppressWarnings("unchecked")
    private void writeProperty(PropertyDescriptor pd, ConfigurationItem ci, WritePasswords writePasswords, ManifestWriter.ManifestCiWriter manifestCiWriter) {
        Object o = pd.get(ci);

        if (o == null) return;
        if (pd.isHidden()) return;
        if (ci.getType().instanceOf(lookupType(ci, Artifact.class)) && ARTIFACT_PROPERTIES_TO_SKIP.contains(pd.getName())) return;

        manifestCiWriter.property(pd.getName());
        switch (pd.getKind()) {
            case BOOLEAN:
            case INTEGER:
            case STRING:
                String value = pd.get(ci).toString();
                if (pd.isPassword()) {
                    value = filterPasswordValue(value, writePasswords);
                }
                manifestCiWriter.writeAsStringValue(value);
                break;
            case ENUM:
                manifestCiWriter.writeAsStringValue(((Enum<?>) pd.get(ci)).name());
                break;
            case DATE:
                Calendar c = Calendar.getInstance();
                c.setTime((Date) o);
                manifestCiWriter.writeAsStringValue(DatatypeConverter.printDateTime(c));
                break;
            case SET_OF_STRING:
            case LIST_OF_STRING:
                manifestCiWriter.writeStringCollectionProperty(ImmutableList.copyOf((Collection<String>) o));
                break;
            case CI:
                manifestCiWriter.writeCiReferenceProperty(cache.lookup((ConfigurationItem) o));
                break;
            case SET_OF_CI:
            case LIST_OF_CI:
                if (pd.isAsContainment()) {
                    writeNestedConfigurationItems((Collection<ConfigurationItem>) o, writePasswords);
                } else {
                    manifestCiWriter.writeCiReferenceCollectionProperty(ImmutableList.copyOf(Collections2.transform((Collection<ConfigurationItem>) o, new Function<ConfigurationItem, String>() {
                        @Override
                        public String apply(ConfigurationItem input) {
                            return cache.lookup(input);
                        }
                    })));
                }
                break;
            case MAP_STRING_STRING:
                manifestCiWriter.writeMapStringStringProperty((Map<String, String>) o);
                break;
        }
        manifestCiWriter.endProperty();
    }

    private void writeNestedConfigurationItems(Collection<ConfigurationItem> cis, WritePasswords writePasswords) {
        for (ConfigurationItem ci : cis) {
            ManifestWriter.ManifestCiWriter ciWriter;

            if (ci instanceof Artifact && isLocalArtifact(ci)) {
                ciWriter = writer.writeCi(cache.lookup(ci), ci.getType(), id(cache.lookup(ci), ((Artifact) ci).getFile().getName()));
            } else {
                ciWriter = writer.writeCi(cache.lookup(ci), ci.getType());
            }
            writeProperties(ci, writePasswords, ciWriter);
            ciWriter.endCi();
        }
    }

    boolean isLocalArtifact(ConfigurationItem ci) {
        if (!ci.getType().instanceOf(lookupType(ci, SourceArtifact.class))) {
            return false;
        }
        if (!ci.hasProperty(SourceArtifact.FILE_URI_PROPERTY_NAME)) {
            return true;
        }
        String fileUri = ci.getProperty(SourceArtifact.FILE_URI_PROPERTY_NAME);
        return Strings.isNullOrEmpty(fileUri) || fileUri.startsWith("jcr:");
    }

    private boolean shouldSkipFileUri(PropertyDescriptor pd, ConfigurationItem ci) {
        return "fileUri".equals(pd.getName()) && isLocalArtifact(ci);
    }

    @Override
    public String toString() {
        return writer.toString();
    }
}
