/**
 * Copyright 2014-2018 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager.io

import java.io.{Closeable, InputStream}
import java.util.jar.JarInputStream
import java.util.zip.{ZipEntry, ZipInputStream}

import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.xldeploy.packager.PackagerConfig
import com.xebialabs.xldeploy.packager.io.StreamerFactory.{ArchiveStreamProvider, JdkArchiveStreamProvider}
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream
import org.apache.commons.compress.archivers.{ArchiveEntry, ArchiveInputStream}
import org.apache.commons.compress.compressors.bzip2.BZip2CompressorInputStream
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream
import org.slf4j.{Logger, LoggerFactory}

import scala.collection.JavaConverters._
import scala.util.Try

sealed trait StreamEntry {
  def getName: String

  def getPath: String

  def getInputStream: InputStream

  def isDirectory: Boolean
}

trait XLArchiveEntry extends StreamEntry {
  def path: String
  def stream: InputStream

  override def getName: String = path.split('/').last

  override def getPath: String = if (isDirectory) {
    path.substring(0, path.length - 1)
  } else {
    path
  }

  override def getInputStream: InputStream = new InputStream {

    override def read(b: Array[Byte]): Int = stream.read(b)

    override def read(b: Array[Byte], off: Int, len: Int): Int = stream.read(b, off, len)

    override def read(): Int = stream.read()

    override def close(): Unit = {}
  }


}

case class ArchivedEntry(ze: ArchiveEntry, stream: ArchiveInputStream) extends XLArchiveEntry {
  override val path: String = ze.getName

  override def isDirectory: Boolean = ze.isDirectory
}

case class ZipArchiveEntry(ze: ZipEntry, stream: ZipInputStream) extends XLArchiveEntry {
  override val path: String = ze.getName

  override def isDirectory: Boolean = ze.isDirectory
}

case class DirectoryEntry(file: OverthereFile, base: String) extends StreamEntry {
  override def getName: String = file.getName

  override def getPath: String = file.getPath.substring(base.length)

  override def getInputStream: InputStream = throw new UnsupportedOperationException("directory does not have getInputStream")

  override def isDirectory: Boolean = true
}

case class FileEntry(file: OverthereFile, base: String) extends StreamEntry {
  override def getName: String = file.getName

  override def getPath: String = file.getPath.substring(base.length)

  override def getInputStream: InputStream = file.getInputStream

  override def isDirectory: Boolean = false
}

case class FileStreamEntry(stream: InputStream, name: String) extends StreamEntry {
  override def getName: String = name

  override def getPath: String = name

  override def getInputStream: InputStream = new InputStream {
    override def read(): Int = stream.read()

    override def read(b: Array[Byte]): Int = stream.read(b)

    override def read(b: Array[Byte], off: Int, len: Int): Int = stream.read(b, off, len)

    override def markSupported(): Boolean = stream.markSupported()

    override def available(): Int = stream.available()

    override def skip(n: Long): Long = stream.skip(n)

    override def reset(): Unit = stream.reset()

    override def close(): Unit = {} // Do not close ourselves, this object does not own the stream

    override def mark(readlimit: Int): Unit = stream.mark(readlimit)
  }

  override def isDirectory = false
}

sealed trait Streamer {
  def stream(): Stream[StreamEntry]
}

object StreamerFactory {
  val logger: Logger = LoggerFactory.getLogger(classOf[StreamerFactory])
  def defaultMappings(): StreamerFactory = forConfig(ConfigFactory.defaultReference())
  def forConfig(config: Config) = new StreamerFactory(new PackagerConfig(config).archiveExtensionMappings)

  type ArchiveStreamProvider = InputStream => ArchiveInputStream
  type JdkArchiveStreamProvider = InputStream => ZipInputStream
}

class StreamerFactory(archiveMappings: Map[String, String]) {

  private[io] def getArchiveEntryStreamer(file: OverthereFile): Streamer = {
    val optionalExtMappingTuple: Option[(String, String)] = archiveMappings.find(p = am => file.getName.endsWith(am._1))
    optionalExtMappingTuple match {
      case None => throw UnsupportedArchiveExtensionException(s"${file.getName} is not a supported archive")
      case Some((originExt, mappedExt)) =>
        StreamerFactory.logger.debug(s"Detected mapped archive extension $originExt -> $mappedExt for ${file.getName}")
        mappedExt match {
          case "zip" => new JdkArchiveStreamer(file, is => new ZipInputStream(is))
          case "jar" => new JdkArchiveStreamer(file, is => new JarInputStream(is))
          case "tar" => new ArchiveStreamer(file, is => new TarArchiveInputStream(is))
          case "tar.gz" => new ArchiveStreamer(file, is => new TarArchiveInputStream(new GzipCompressorInputStream(is)))
          case "tar.bz2" => new ArchiveStreamer(file, is => new TarArchiveInputStream(new BZip2CompressorInputStream(is)))
          case _ => throw UnsupportedArchiveExtensionException(s"${file.getName} with extension $originExt -> $mappedExt is not a supported archive")
      }
    }
  }

  private[io] def getArchiveEntryStreamStreamer(is: InputStream, name: String): Streamer = {
    val optionalExtMappingTuple: Option[(String, String)] = archiveMappings.find(p = am => name.endsWith(am._1))
    optionalExtMappingTuple match {
      case None => throw UnsupportedArchiveExtensionException(s"$name is not a supported archive")
      case Some((originExt, mappedExt)) =>
        StreamerFactory.logger.debug(s"Detected mapped archive extension $originExt -> $mappedExt for $name")
        mappedExt match {
          case "zip" => new JdkArchiveStreamStreamer(is, is => new ZipInputStream(is))
          case "jar" => new JdkArchiveStreamStreamer(is, is => new JarInputStream(is))
          case "tar" => new ArchiveStreamStreamer(is, is => new TarArchiveInputStream(is))
          case "tar.gz" => new ArchiveStreamStreamer(is, is => new TarArchiveInputStream(new GzipCompressorInputStream(is)))
          case "tar.bz2" => new ArchiveStreamStreamer(is, is => new TarArchiveInputStream(new BZip2CompressorInputStream(is)))
          case _ => throw UnsupportedArchiveExtensionException(s"$name with extension $originExt -> $mappedExt is not a supported archive")
        }
    }
  }

  def isArchive(name: String): Boolean = {
    val optionalExtMappingTuple: Option[(String, String)] = archiveMappings.find(p = am => name.endsWith(am._1))
    optionalExtMappingTuple.isDefined
  }

  def streamer(is: InputStream, name: String): Streamer = {
    if (isArchive(name)) {
      getArchiveEntryStreamStreamer(is, name)
    } else {
      new FileStreamStreamer(is, name)
    }
  }

  def streamer(file: OverthereFile): Streamer = {
    if (file.isDirectory) {
      new DirectoryStreamer(file, this)
    } else if (isArchive(file.getName)) {
      getArchiveEntryStreamer(file)
    } else {
      new FileStreamer(file)
    }
  }
}

class DirectoryStreamer(dir: OverthereFile, sf: StreamerFactory) extends Streamer {
  override def stream(): Stream[StreamEntry] = {
    def fileStream(dir: OverthereFile, base: String): Stream[StreamEntry] =
      if (dir.isDirectory)
        Option(dir.listFiles)
          .map(_.asScala.toList.sortBy(_.getPath).toStream.flatMap(file =>
            if (file.isDirectory) {
              DirectoryEntry(file, base) #:: fileStream(file, base)
            } else {
              FileEntry(file, base) #:: fileStream(file, base)
            }))
          .getOrElse(Stream.empty)
      else Stream.empty

    val base = dir.getPath + "/"
    fileStream(dir, base)
  }
}

class FileStreamer(file: OverthereFile) extends Streamer {
  override def stream(): Stream[StreamEntry] = FileEntry(file, file.getParentFile.getPath + "/") #:: Stream.empty[StreamEntry]
}

class FileStreamStreamer(is: InputStream, name: String) extends Streamer {
  override def stream(): Stream[StreamEntry] = FileStreamEntry(is, name) #:: Stream.empty[StreamEntry]
}

trait ArchiveEntryStreamer extends Streamer {
  def nextEntry(s: ArchiveInputStream, toClose: Set[Closeable]): Stream[StreamEntry] = {
    Option(s.getNextEntry) match {
      case None =>
        toClose.foreach(c => Try(c.close()))
        Stream.empty
      case Some(e) => ArchivedEntry(e, s) #:: nextEntry(s, toClose)
    }
  }
}

trait ZipEntryStreamer extends Streamer {
  def nextEntry(s: ZipInputStream, toClose: Set[Closeable]): Stream[StreamEntry] = {
    Option(s.getNextEntry) match {
      case None =>
        toClose.foreach(c => Try(c.close()))
        Stream.Empty
      case Some(e) => ZipArchiveEntry(e, s) #:: nextEntry(s, toClose)
    }
  }
}

class JdkArchiveStreamer(file: OverthereFile, streamProvider: JdkArchiveStreamProvider) extends ZipEntryStreamer {
  override def stream(): Stream[StreamEntry] = {
    val is = file.getInputStream
    val jdkIs = streamProvider(is)
    nextEntry(jdkIs, Set(is, jdkIs))
  }
}

class JdkArchiveStreamStreamer(is: InputStream, streamProvider: JdkArchiveStreamProvider) extends ZipEntryStreamer {
  override def stream(): Stream[StreamEntry] = {
    val jdkIs = streamProvider(is)
    nextEntry(jdkIs, Set())
  }
}

class ArchiveStreamer(file: OverthereFile, val streamProvider: ArchiveStreamProvider) extends ArchiveEntryStreamer {
  override def stream(): Stream[StreamEntry] = {
    val is = file.getInputStream
    val archiveStream = streamProvider(is)
    nextEntry(archiveStream, Set(is, archiveStream))
  }
}

class ArchiveStreamStreamer(is: InputStream, val streamProvider: ArchiveStreamProvider) extends ArchiveEntryStreamer {
  override def stream(): Stream[StreamEntry] = {
    val archiveStream = streamProvider(is)
    nextEntry(archiveStream, Set())
  }
}

case class UnsupportedArchiveExtensionException(str: String) extends RuntimeException(str)