/**
 * Copyright 2014-2018 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager

import java.io._
import java.nio.charset.StandardCharsets
import java.security.{DigestInputStream, MessageDigest}
import java.util.regex.Pattern
import java.util.regex.Pattern.{CASE_INSENSITIVE, COMMENTS}

import com.xebialabs.deployit.plugin.api.udm.artifact.{FileArtifact, FolderArtifact, SourceArtifact}
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployableArtifact.SCAN_PLACEHOLDERS_PROPERTY_NAME
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.xldeploy.packager.SourceArtifactEnricher._
import com.xebialabs.xldeploy.packager.io.{DevNull, StreamEntry, StreamerFactory}
import com.xebialabs.xldeploy.packager.utils.TryWith
import org.apache.commons.codec.binary.Hex
import org.slf4j.{Logger, LoggerFactory}

import scala.collection.JavaConverters._

object SourceArtifactEnricher {
  private[SourceArtifactEnricher] val logger: Logger = LoggerFactory.getLogger(classOf[SourceArtifactEnricher])
  val defaultDelims = "{{ }}"
  var patternMap: Map[String, Pattern] = Map()
  val devNull = new DevNull
  val devNullWriter = new OutputStreamWriter(devNull)

  def compilePattern(regex: String): Pattern = patternMap.get(regex) match {
    case Some(compiled) => compiled
    case None =>
      val compiled = Pattern.compile(regex, COMMENTS | CASE_INSENSITIVE)
      patternMap += (regex -> compiled)
      compiled
  }

  def hasCheckSum(bda: SourceArtifact): Boolean = {
    bda.getChecksum != null && !bda.getChecksum.isEmpty
  }

  def getDelimiters(artifact: SourceArtifact): String = if (artifact.hasProperty("delimiters")) {
    artifact.getProperty[String]("delimiters")
  } else {
    defaultDelims
  }


  def shouldScanPlaceholders(artifact: SourceArtifact): Boolean = {
    !artifact.hasProperty(SCAN_PLACEHOLDERS_PROPERTY_NAME) || artifact.getProperty[Boolean](SCAN_PLACEHOLDERS_PROPERTY_NAME)
  }
}

case class StreamFile(name: String, inputStream: InputStream)

class SourceArtifactEnricher(streamerFactory: StreamerFactory) {

  def enrichArtifact(sa: SourceArtifact): Unit = {
    enrichArtifact(sa, None, devNull)
  }

  def enrichArtifact(sa: SourceArtifact, streamFile: StreamFile): Unit = {
    enrichArtifact(sa, Option(streamFile), devNull)
  }

  def enrichArtifact(sa: SourceArtifact, streamFile: Option[StreamFile], os: OutputStream): Unit = {
    val digest: Option[MessageDigest] = if (!hasCheckSum(sa)) Some(MessageDigest.getInstance("SHA1")) else None
    val mustacher = Mustacher(SourceArtifactEnricher.getDelimiters(sa))

    sa match {
      case sa: SourceArtifact if !shouldScanPlaceholders(sa) && hasCheckSum(sa) =>
        logger.info(s"Not enriching $sa as no placeholders should be scanned and checksum is present")
      case fa: FolderArtifact =>
        logger.info(s"Going to scan artifact $fa as a folder artifact")
        scanFolderArtifact(fa, streamFile, digest, mustacher)
      case _ if streamFile.isEmpty =>
        logger.info(s"Going to scan artifact $sa")
        TryWith(sa.getFile.getInputStream) { fis =>
          scanFileOrArchiveArtifact(sa, StreamFile(sa.getFile.getName, fis), os, digest, mustacher)
        }
      case _ =>
        logger.info(s"Going to scan artifact $sa from stream")
        scanFileOrArchiveArtifact(sa, streamFile.get, os, digest, mustacher)
    }

    if (shouldScanPlaceholders(sa)) {
      sa.setPlaceholders(mustacher.placeholders.asJava)
    }
    digest.foreach(d => sa.setProperty(SourceArtifact.CHECKSUM_PROPERTY_NAME, Hex.encodeHexString(d.digest())))
  }

  private def scanFolderArtifact(fa: FolderArtifact, streamFile: Option[StreamFile], digest: Option[MessageDigest], mustacher: Mustacher): Unit = {
    val textFileRegex = compilePattern(fa.getTextFileNamesRegex)
    val excludePathRegex = Option(fa.getExcludeFileNamesRegex).filterNot(_.isEmpty).map(compilePattern)
    val scanPlaceholders = shouldScanPlaceholders(fa)
    val streamer = streamFile match {
      case Some(x) => streamerFactory.streamer(x.inputStream, x.name)
      case None => streamerFactory.streamer(fa.getFile)
    }
    streamer.stream().foreach { entry =>
      var digestPath = entry.getPath + (if (entry.isDirectory) File.separator else "")
      digestPath = digestPath.replace("\\", "/")
      logger.trace(s"scanFolderArtifact: Digest path [$digestPath]")
      digest.foreach(d => d.update(digestPath.getBytes(StandardCharsets.UTF_8)))
      if (entry.isDirectory) {
        //skip
      } else {
        TryWith(entry.getInputStream) { is =>
          val inputStream = digest.map(new DigestInputStream(is, _)).getOrElse(is)
          if (scanPlaceholders && !excludePathRegex.exists(_.matcher(entry.getPath).matches()) && !entry.isDirectory) {
            if (streamerFactory.isArchive(entry.getName)) {
              streamerFactory.streamer(entry.getInputStream, entry.getName).stream().foreach { e2 =>
                scanEntry(fa, e2, textFileRegex, excludePathRegex, mustacher)
              }
            } else if (textFileRegex.matcher(entry.getName).matches()) {
              // TODO character sets (+ BOM detection?)
              TryWith(mustacher.newReader(new InputStreamReader(inputStream))) { r =>
                OverthereUtils.write(r, devNullWriter)
              }
            } else {
              OverthereUtils.write(inputStream, devNull)
            }
          } else {
            OverthereUtils.write(inputStream, devNull)
          }
        }
      }
    }
  }

  private[this] def scanFileOrArchiveArtifact(fa: SourceArtifact, streamFile: StreamFile, os: OutputStream, digest: Option[MessageDigest], mustacher: Mustacher): Unit = {
    val textFileRegex = compilePattern(fa.getTextFileNamesRegex)
    val excludePathRegex = Option(fa.getExcludeFileNamesRegex).filterNot(_.isEmpty).map(compilePattern)
    val inputStream = digest.map(new DigestInputStream(streamFile.inputStream, _)).getOrElse(streamFile.inputStream)
    if (shouldScanPlaceholders(fa)) {
      streamerFactory.streamer(inputStream, streamFile.name).stream().foreach { entry =>
        scanEntry(fa, entry, textFileRegex, excludePathRegex, mustacher)
      }
      // Ensure that we empty the stream so that all bytes are read, testcases seem to prove this is not needed,
      // But we don't want to accidentally calculate wrong checksums.
      OverthereUtils.write(inputStream, os)
    } else {
      logger.debug(s"Artifact [$fa] has disabled placeholder scanning")
      OverthereUtils.write(inputStream, os)
    }
  }

  private[this] def scanEntry(fa: SourceArtifact, entry: StreamEntry, textFileRegex: Pattern, excludePathRegex: Option[Pattern], mustacher: Mustacher): Unit = {
    if (!entry.isDirectory && !excludePathRegex.exists(_.matcher(entry.getPath).matches())) {
      if (textFileRegex.matcher(entry.getName).matches()) {
        logger.debug(s"$fa: ${entry.getName} is a text file.")
        // TODO character sets (+ BOM detection?)
        TryWith(mustacher.newReader(new InputStreamReader(entry.getInputStream))) { r =>
          OverthereUtils.write(r, devNullWriter)
        }
      } else if (streamerFactory.isArchive(entry.getName)) {
        logger.debug(s"$fa: Detected archive for ${entry.getName}")
        streamerFactory.streamer(entry.getInputStream, entry.getName).stream().foreach { e2 =>
          scanEntry(fa, e2, textFileRegex, excludePathRegex, mustacher)
        }
      }
    } else {
      logger.debug(s"$fa: Skipping ${entry.getName}")
    }
  }
}
