/**
 * Copyright 2014-2018 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.util;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;

public enum BOM {
    NONE("NONE", new byte[] {}, null),
    UTF8("UTF-8", new byte[] { (byte) 0xEF, (byte) 0xBB, (byte) 0xBF }, StandardCharsets.UTF_8),
    UTF16BE("UTF-16BE", new byte[] { (byte) 0xFE, (byte) 0xFF }, StandardCharsets.UTF_16BE),
    UTF16LE("UTF-16LE", new byte[] { (byte) 0xFF, (byte) 0xFE }, StandardCharsets.UTF_16LE);

    private final String name;
    private final byte[] bom;
    private Charset charset;

    BOM(String encoding, byte[] bom, Charset charset) {
        this.name = encoding;
        this.bom = bom;
        this.charset = charset;
    }

    public static BOM matchedBOM(byte[] bomToMatch) {

        if (bomToMatch == null) {
            throw new IllegalArgumentException("BOM bytes cannot be null");
        }

        for (BOM bom : values()) {
            byte[] bomBytes = bom.getBom();
            if (bomBytes.length > 0 && Arrays.equals(bomBytes, Arrays.copyOfRange(bomToMatch, 0, bomBytes.length))) {
                return bom;
            }
        }

        return NONE;
    }

    public String getName() {
        return name;
    }

    public boolean isDefaultCharset() {
        return charset == null;
    }

    public Charset getCharset() {
        return charset;
    }

    public int getSize() {
        return bom.length;
    }

    public byte[] getBom() {
        return bom.clone();
    }

    public String toString() {
        return name;
    }

}
