/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager

import java.io._
import java.nio.charset.{Charset, StandardCharsets}
import java.security.{DigestInputStream, MessageDigest}
import java.util.regex.Pattern
import java.util.regex.Pattern.{CASE_INSENSITIVE, COMMENTS}

import com.xebialabs.deployit.io.LocalFileBased
import com.xebialabs.deployit.plugin.api.udm.artifact.{DerivedArtifact, FolderArtifact, SourceArtifact}
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployableArtifact.SCAN_PLACEHOLDERS_PROPERTY_NAME
import com.xebialabs.deployit.util.{BOM, DetectBOM, TryWith}
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.xldeploy.packager.SourceArtifactEnricher._
import com.xebialabs.xldeploy.packager.io._
import org.apache.commons.codec.binary.Hex
import org.apache.commons.compress.archivers.jar.JarArchiveEntry
import org.apache.commons.compress.archivers.tar.TarArchiveEntry
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry
import org.apache.commons.compress.archivers.{ArchiveEntry, ArchiveOutputStream}
import org.apache.commons.io.IOUtils
import org.slf4j.{Logger, LoggerFactory}

import scala.collection.JavaConverters._
import scala.language.postfixOps
import scala.util.{Failure, Success}

object SourceArtifactEnricher {
  private[SourceArtifactEnricher] val logger: Logger = LoggerFactory.getLogger(classOf[SourceArtifactEnricher])

  private[SourceArtifactEnricher] sealed trait ScanType

  private[SourceArtifactEnricher] case object DigestOnly extends ScanType

  private[SourceArtifactEnricher] case object ProcessArchive extends ScanType

  private[SourceArtifactEnricher] case object ProcessTextFile extends ScanType

  private[SourceArtifactEnricher] case class ScanSpec(scanPlaceholders: Boolean, excludePathRegex: Option[Pattern], textFileRegex: Pattern) {
    def this(fa: SourceArtifact) = this(
      shouldScanPlaceholders(fa),
      Option(fa.getExcludeFileNamesRegex).filterNot(_.isEmpty).map(compilePattern),
      compilePattern(fa.getTextFileNamesRegex)
    )
  }

  implicit class ScanHandler(scanSpec: ScanSpec)(implicit streamerFactory: StreamerFactory) {
    def getScanType(entry: StreamEntry): ScanType = {
      if (!scanSpec.scanPlaceholders) DigestOnly
      else if (scanSpec.excludePathRegex.exists(_.matcher(entry.getPath).matches())) DigestOnly
      else if (streamerFactory.hasArchiveExtension(entry.getName)) ProcessArchive
      else if (scanSpec.textFileRegex.matcher(entry.getName).matches()) ProcessTextFile
      else DigestOnly
    }
  }

  val defaultDelims = "{{ }}"
  var patternMap: Map[String, Pattern] = Map()
  val devNull = new DevNull
  val devNullWriter = new OutputStreamWriter(devNull)

  private[this] def compilePattern(regex: String): Pattern = patternMap.get(regex) match {
    case Some(compiled) => compiled
    case None =>
      val compiled = Pattern.compile(regex, COMMENTS | CASE_INSENSITIVE)
      patternMap += (regex -> compiled)
      compiled
  }

  private[SourceArtifactEnricher] def hasCheckSum(bda: SourceArtifact): Boolean = {
    bda.getChecksum != null && !bda.getChecksum.isEmpty
  }

  private[SourceArtifactEnricher] def getDelimiters(artifact: SourceArtifact): String = if (artifact.hasProperty("delimiters")) {
    artifact.getProperty[String]("delimiters")
  } else {
    defaultDelims
  }


  private[SourceArtifactEnricher] def shouldScanPlaceholders(artifact: SourceArtifact): Boolean = {
    !artifact.hasProperty(SCAN_PLACEHOLDERS_PROPERTY_NAME) || artifact.getProperty[Boolean](SCAN_PLACEHOLDERS_PROPERTY_NAME)
  }

  private[SourceArtifactEnricher] def shouldScanArtifactPlaceholders(artifact: SourceArtifact, da: DerivedArtifact[_ <: SourceArtifact]): Boolean = {
    shouldScanPlaceholders(artifact) && da.getPlaceholders != null && da.getPlaceholders.size() > 0
  }

}

case class StreamFile(name: String, inputStream: InputStream)

class SourceArtifactEnricher(streamerFactory: StreamerFactory) {
  implicit val sf: StreamerFactory = streamerFactory

  def enrichArtifact(sa: SourceArtifact): Unit = {
    enrichArtifact(sa, None, devNull)
  }

  def enrichArtifact(sa: SourceArtifact, streamFile: StreamFile): Unit = {
    enrichArtifact(sa, Option(streamFile), devNull)
  }

  def enrichArtifact(sa: SourceArtifact, streamFile: Option[StreamFile], os: OutputStream): Unit = {
    val digest: Option[MessageDigest] = if (!hasCheckSum(sa)) Some(MessageDigest.getInstance("SHA1")) else None

    if (shouldScanPlaceholders(sa) || !hasCheckSum(sa)) {
      val mustacher = Mustacher(getDelimiters(sa))

      sa match {
        case fa: FolderArtifact =>
          logger.info(s"Going to scan artifact $fa as a folder artifact")
          scanFolderArtifact(fa, streamFile, digest, mustacher)
        case _ if streamFile.isEmpty =>
          logger.info(s"Going to scan artifact $sa")
          TryWith(sa.getFile.getInputStream) { fis =>
            scanFileOrArchiveArtifact(sa, StreamFile(sa.getFile.getName, fis), os, digest, mustacher)
          }
        case _ =>
          logger.info(s"Going to scan artifact $sa from stream")
          scanFileOrArchiveArtifact(sa, streamFile.get, os, digest, mustacher)
      }

      sa.setPlaceholders(mustacher.placeholders.asJava)
      digest.foreach(d => sa.setProperty(SourceArtifact.CHECKSUM_PROPERTY_NAME, Hex.encodeHexString(d.digest())))
    } else {
      logger.info(s"Not enriching $sa as no placeholders should be scanned and checksum is present")
    }
  }

  def enrichArtifact(da: DerivedArtifact[_ <: SourceArtifact]): Unit = {
    if (da.getSourceArtifact == null) {
      da.setFile(null)
    } else {
      val derivedFile = createDerivedFile(da)
      da.setFile(derivedFile)
    }
  }

  def createDerivedFile(da: DerivedArtifact[_ <: SourceArtifact]): OverthereFile = {
    val sa = da.getSourceArtifact
    val sourceFile = da.getSourceArtifact.getFile
    val workDir = sourceFile.getParentFile
    val derivedFile = OverthereUtils.getUniqueFolder(workDir, sa.getName).getFile(sourceFile.getName)
    sourceFile.copyTo(derivedFile)

    replacePlaceholdersInArtifact(da, derivedFile)
    derivedFile
  }

  private[this] def updateDigestWithFilename(entry: StreamEntry, digest: MessageDigest): Unit = {
    var digestPath = entry.getPath + (if (entry.isDirectory) File.separator else "")
    digestPath = digestPath.replace("\\", "/")
    logger.trace(s"scanFolderArtifact: Digest path [$digestPath]")
    digest.update(digestPath.getBytes(StandardCharsets.UTF_8))
  }

  private[this] def scanFolderArtifact(fa: FolderArtifact, streamFile: Option[StreamFile], digest: Option[MessageDigest],
                                       mustacher: Mustacher): Unit = {
    val streamer = streamFile match {
      case Some(x) => streamerFactory.streamer(x.inputStream, x.name)
      case None => streamerFactory.streamer(toLocalFile(fa))
    }
    streamer.stream().foreach { entry =>
      digest.foreach(d => updateDigestWithFilename(entry, d))
      if (!entry.isDirectory) {
        doScan(fa, entry, mustacher, is => digest.map(new DigestInputStream(is, _)).getOrElse(is))
      }
    }
  }

  private[this] def scanFileOrArchiveArtifact(sa: SourceArtifact, streamFile: StreamFile, os: OutputStream,
                                              digest: Option[MessageDigest], mustacher: Mustacher): Unit = {
    val inputStream = digest.map(new DigestInputStream(streamFile.inputStream, _)).getOrElse(streamFile.inputStream)
    if (shouldScanPlaceholders(sa)) {
      streamerFactory.streamer(inputStream, streamFile.name).stream().foreach { entry =>
        if (!entry.isDirectory) {
          doScan(sa, entry, mustacher, identity[InputStream])
        }
      }
      // Ensure that we empty the stream so that all bytes are read, testcases seem to prove this is not needed,
      // But we don't want to accidentally calculate wrong checksum.
      OverthereUtils.write(inputStream, os)
    } else {
      logger.debug(s"Artifact [$sa] has disabled placeholder scanning")
      OverthereUtils.write(inputStream, os)
    }
  }

  private[this] def doScan(sa: SourceArtifact, entry: StreamEntry, mustacher: Mustacher,
                           transform: InputStream => InputStream): Unit = {

    val scanSpec = new ScanSpec(sa)
    if (!entry.isDirectory) {
      TryWith(entry.getInputStream) { inputStream =>
        val is = transform(inputStream)

        scanSpec.getScanType(entry) match {
          case DigestOnly =>
            logger.debug(s"$sa: Skipping ${entry.getName}")
            OverthereUtils.write(is, devNull) // Just write the bytes

          case ProcessArchive =>
            logger.debug(s"$sa: Detected archive for ${entry.getName}")
            streamerFactory.streamer(is, entry.getName).stream().foreach { e2 =>
              if (!e2.isDirectory) {
                doScan(sa, e2, mustacher, identity[InputStream])
              }
            }
            // Ensure that we empty the stream so that all bytes are read, so that we don't accidentally calculate wrong checksum.
            OverthereUtils.write(is, devNull) // Finish the archive

          case ProcessTextFile =>
            logger.debug(s"$sa: ${entry.getName} is a text file.")
            val resettableInputStream = ArtifactIOUtils.getResettableInputStream(is)
            val reader: InputStreamReader = DetectBOM.detect(resettableInputStream) match {
              case BOM.NONE => new InputStreamReader(resettableInputStream)
              case bom: BOM => new InputStreamReader(resettableInputStream, bom.getCharset)
            }
            TryWith(mustacher.newReader(reader)) { r =>
              OverthereUtils.write(r, devNullWriter) // Scan the file for placeholders using the Mustacher
            }
        }
      }
    }
  }

  private[this] def replacePlaceholdersInArtifact(da: DerivedArtifact[_ <: SourceArtifact], derivedFile: OverthereFile): Unit = {
    val sa = da.getSourceArtifact
    if (shouldScanArtifactPlaceholders(sa, da)) {
      val mustacher = MustacherReplacer(getDelimiters(sa))
      val scanSpec = new ScanSpec(sa)

      sa match {
        case fa: FolderArtifact =>
          logger.info(s"Going to replace placeholders in artifact $fa as in a folder artifact")
          replacePlaceholdersFolderArtifact(da, derivedFile, mustacher, scanSpec)

        case _ if streamerFactory.hasArchiveExtension(sa.getFile.getName) =>
          logger.info(s"Going to replace placeholders in artifact $sa as in an archive artifact")
          replacePlaceholdersArchiveArtifact(da, derivedFile, mustacher, scanSpec)

        case _ =>
          logger.info(s"Going to replace placeholders in artifact $sa from stream")
          replacePlaceholdersFileArtifact(da, derivedFile, mustacher, scanSpec)
      }
    } else {
      logger.info(s"Not enriching $sa as no placeholders should be scanned")
    }
  }

  private[this] def replacePlaceholdersFolderArtifact(sa: DerivedArtifact[_ <: SourceArtifact], derivedFile: OverthereFile,
                                                      mustacher: MustacherReplacer, scanSpec: ScanSpec): Unit = {
    val streamer = streamerFactory.streamer(toLocalFile(sa.getSourceArtifact))
    streamer.stream().foreach { entry =>
      doReplacePlaceholders(sa, derivedFile, derivedFile.getPath, entry, mustacher, scanSpec)
    }
  }

  private[this] def replacePlaceholdersFileArtifact(sa: DerivedArtifact[_ <: SourceArtifact], derivedFile: OverthereFile,
                                                    mustacher: MustacherReplacer, scanSpec: ScanSpec): Unit = {
    val streamer = streamerFactory.streamer(toLocalFile(sa.getSourceArtifact))
    streamer.stream().foreach { entry =>
      doReplacePlaceholders(sa, derivedFile, derivedFile.getPath, entry, mustacher, scanSpec)
    }
  }

  private[this] def replacePlaceholdersArchiveArtifact(da: DerivedArtifact[_ <: SourceArtifact], derivedFile: OverthereFile,
                                                       mustacher: MustacherReplacer, scanSpec: ScanSpec): Unit = {
    val file = new File(derivedFile.getPath)
    val parentFile = new File(derivedFile.getParentFile.getPath)
    val sa = da.getSourceArtifact
    withArchiveOutputStream(file, os => {
      streamerFactory.streamer(toLocalFile(sa)).stream().foreach {
        case e: XLArchiveEntry =>
          logger.debug(s"$sa: ${e.getName} is an archive entry file.")
          processEntry(e.getInputStream, os, parentFile, e, mustacher, da.getPlaceholders.asScala.toMap, scanSpec, sa)
      }
    })
  }

  private def toLocalFile(sa: SourceArtifact): File = sa.getFile match {
    case l: LocalFile => l.getFile
    case s: LocalFileBased => s.getLocalFile
    case x => throw new RuntimeException(s"Source artifact ${sa.getName} is not local file based but ${x.getClass}")
  }

  private def doReplacePlaceholders(da: DerivedArtifact[_ <: SourceArtifact], derivedFile: OverthereFile, basePath: String, entry: StreamEntry, mustacher: MustacherReplacer, scanSpec: ScanSpec): Unit = {
    if (!entry.isDirectory) {
      TryWith(entry.getInputStream) { is =>
        val sa = da.getSourceArtifact
        val baseFile = new File(basePath)
        scanSpec.getScanType(entry) match {
          case DigestOnly =>
            logger.debug(s"$da: Skipping ${entry.getName}")
            val target = new File(derivedFile.getPath, entry.getPath)
            TryWith(new FileOutputStream(target)) { os =>
              OverthereUtils.write(is, os) // Just write the bytes
            }

          case ProcessArchive if sa.isInstanceOf[FolderArtifact] =>
            logger.debug(s"$da: Detected archive for ${entry.getName} in source Folder artifact $sa")
            val target = new File(derivedFile.getPath, entry.getPath)
            withArchiveOutputStream(target, os => {
              streamerFactory.streamer(entry.getInputStream, entry.getName).stream().foreach {
                case e: XLArchiveEntry =>
                  processEntry(e.getInputStream, os, baseFile, e, mustacher, da.getPlaceholders.asScala.toMap, scanSpec, sa)
              }
            })

          case ProcessArchive =>
            logger.debug(s"$da: Detected archive for ${entry.getName} in source artifact $sa")
            withArchiveOutputStream(new File(derivedFile.getPath), os => {
              processEntry(is, os, baseFile, entry.asInstanceOf[XLArchiveEntry], mustacher, da.getPlaceholders.asScala.toMap, scanSpec, sa)
            })

          case ProcessTextFile =>
            logger.debug(s"$da: ${entry.getName} is a text file.")
            doReplacePlaceholdersTextFile(da, derivedFile, baseFile, is, entry, mustacher)
        }
      }
    }
  }

  private[this] def doReplacePlaceholdersTextFile(sa: DerivedArtifact[_ <: SourceArtifact], derivedFile: OverthereFile, baseFile: File,
                                                  is: InputStream, entry: StreamEntry, mustacher: MustacherReplacer): Unit = {
    val resettableInputStream = ArtifactIOUtils.getResettableInputStream(is)

    val charset = DetectBOM.detect(resettableInputStream) match {
      case BOM.NONE => getCharset(sa.getSourceArtifact, entry)
      case bom: BOM => Option(bom.getCharset)
    }

    val reader = charset.map(cs => new InputStreamReader(resettableInputStream, cs)).getOrElse(new InputStreamReader(resettableInputStream))

    val targetPath = if (derivedFile.isDirectory) {
      new File(baseFile, entry.getPath)
    } else {
      baseFile
    }
    val wrappedWriter = charset.map(cs => new OutputStreamWriter(new FileOutputStream(targetPath), cs.newEncoder())).getOrElse(new OutputStreamWriter(new FileOutputStream(targetPath)))

    TryWith(mustacher.newReader(reader, sa.getPlaceholders.asScala.toMap)) { r =>
      TryWith(wrappedWriter) { w =>
        OverthereUtils.write(r, w)
      }
    }
  }

  private[this] def processEntry(is: InputStream, os: ArchiveOutputStream, baseFile: File, entry: XLArchiveEntry, mustacher: MustacherReplacer,
                                 placeholders: Map[String, String], scanSpec: ScanSpec, sa: SourceArtifact): Unit = {
    val entryName = entry.getName

    scanSpec.getScanType(entry) match {
      case ProcessArchive =>
        processArchive()
      case ProcessTextFile if !entry.isDirectory =>
        processTextFile()
      case _ =>
        processOtherFileOrFolder()
    }

    def processArchive(): Unit = {
      logger.debug(s"Replacing placeholders in $entryName for path ${entry.getPath} as in an archive artifact")
      val now = System.currentTimeMillis()
      val inFile = new File(baseFile, s"in-$now-$entryName")
      val outFile = new File(baseFile, s"out-$now-$entryName")
      TryWith(new FileOutputStream(inFile)) { IOUtils.copy(entry.getInputStream, _) }
      withArchiveOutputStream(outFile, temporaryArchiveOutputStream => {
        streamerFactory.streamer(inFile).stream().foreach {
          case inner: XLArchiveEntry =>
            logger.debug(s"Processing entry [${inner.getName}] with path [${inner.getPath}] for parent entry [$entryName]")
            processEntry(inner.getInputStream, temporaryArchiveOutputStream, baseFile, inner, mustacher, placeholders, scanSpec, sa)
        }
      })
      logger.debug(s"Processed entry [$entryName]")
      val ze = entry match {
        case e: JarArchivedEntry => new JarArchiveEntry(e.ze)
        case e: ZipArchivedEntry => new ZipArchiveEntry(e.ze)
        case e: ArchivedEntry => new TarArchiveEntry(outFile, e.ze.getName)
        case e => e.ze
      }
      withOpenArchiveEntry(os, ze, () => {
        TryWith(new FileInputStream(outFile)) { is =>
          OverthereUtils.write(is, os)
        } match {
          case Success(_) =>
            logger.debug(s"Copied data to the parent archive for entry [$entryName]")
            inFile.delete()
            outFile.delete()
            logger.debug(s"Deleted temporary data")
          case Failure(err) =>
            logger.error(s"Could not copy data from temporary file")
            logger.error(err.getMessage, err)
        }
      })
    }

    def processTextFile(): Unit = {
      logger.debug(s"Replacing placeholders in $entryName for path ${entry.getPath}")
      val resettableInputStream = ArtifactIOUtils.getResettableInputStream(is)
      val charset = DetectBOM.detect(resettableInputStream) match {
        case BOM.NONE => getCharset(sa, entry)
        case bom: BOM => Option(bom.getCharset)
      }

      logger.debug(s"Entry $entryName has encoding: ${charset.getOrElse("Unknown")}")

      val reader = charset.map(cs => new InputStreamReader(resettableInputStream, cs)).getOrElse(new InputStreamReader(resettableInputStream))

      val (ze, convertedInputStream, processed) = entry match {
        case e: JarArchivedEntry =>
          (new JarArchiveEntry(e.ze), resettableInputStream, false)
        case e: ZipArchivedEntry =>
          (new ZipArchiveEntry(e.ze), resettableInputStream, false)
        case e: ArchivedEntry =>
          val wrappedReader = mustacher.newReader(reader, placeholders)
          val tae = new TarArchiveEntry(e.ze.getName)
          val buf = charset.map(cs => IOUtils.toByteArray(wrappedReader, cs)).getOrElse(IOUtils.toByteArray(wrappedReader, Charset.defaultCharset()))
          tae.setSize(buf.size)
          (tae, new ByteArrayInputStream(buf), true)
        case e =>
          (e.ze, resettableInputStream, false)
      }

      withOpenArchiveEntry(os, ze, () => {
        val convertedReader = charset.map(cs => new InputStreamReader(convertedInputStream, cs)).getOrElse(new InputStreamReader(convertedInputStream))
        val wrappedReader = if (processed) {
          convertedReader
        } else {
          mustacher.newReader(convertedReader, placeholders)
        }
        copyBytes(wrappedReader, os, charset)
      })
    }

    def processOtherFileOrFolder(): Unit = { // other files and folders, we cannot skip folders because those can be used
      logger.debug(s"Processing $entryName for path ${entry.getPath}")
      val ze = entry match {
        case e: JarArchivedEntry => new JarArchiveEntry(e.ze)
        case e: ZipArchivedEntry => new ZipArchiveEntry(e.ze)
        case e: ArchivedEntry => e.ze
      }
      withOpenArchiveEntry(os, ze, () => {
        OverthereUtils.write(is, os)
      })
    }
  }

  private[this] def withOpenArchiveEntry(os: ArchiveOutputStream, ae: ArchiveEntry, process: () => Unit): Unit = {
    os.putArchiveEntry(ae)
    process()
    os.closeArchiveEntry()
  }

  private def withArchiveOutputStream(file: File, block: ArchiveOutputStream => Unit): Unit = {
    val os = streamerFactory.outputStream(file)
    block(os)
    os.finish()
    os.close()
  }

  private[this] def copyBytes(wrappedReader: Reader, os: OutputStream, maybeCharset: Option[Charset]): Unit = {
    val bytes = maybeCharset.map(cs => IOUtils.toByteArray(wrappedReader, cs)).getOrElse(IOUtils.toByteArray(wrappedReader, Charset.defaultCharset()))
    os.write(bytes)
  }

  private[this] def getCharset(artifact: SourceArtifact, file: StreamEntry): Option[Charset] = {
    val fileEncodings = artifact.getFileEncodings
    for (regexToCharset <- fileEncodings.entrySet.asScala) {
      if (file.getPath.matches(regexToCharset.getKey)) {
        logger.debug("Relative path [{}] matched regex [{}], using charset [{}]", file.getPath, regexToCharset.getKey, regexToCharset.getValue)
        return Option(Charset.forName(regexToCharset.getValue))
      }
    }
    None
  }
}
