/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager.io

import java.io._
import java.util.{Arrays => JArrays, Enumeration => JEnumeration}

import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.xldeploy.packager.PackagerConfig
import com.xebialabs.xldeploy.packager.io.StreamerFactory.{ArchiveStreamProvider, JarArchiveStreamProvider,
  ZipArchiveStreamProvider}
import org.apache.commons.compress.archivers.jar.JarArchiveInputStream
import org.apache.commons.compress.archivers.tar.{TarArchiveInputStream, TarArchiveOutputStream}
import org.apache.commons.compress.archivers.zip._
import org.apache.commons.compress.archivers.{ArchiveEntry, ArchiveInputStream, ArchiveOutputStream, ArchiveStreamFactory}
import org.apache.commons.compress.compressors.CompressorStreamFactory
import org.apache.commons.compress.compressors.bzip2.{BZip2CompressorInputStream, BZip2CompressorOutputStream}
import org.apache.commons.compress.compressors.gzip.{GzipCompressorInputStream, GzipCompressorOutputStream}
import org.slf4j.{Logger, LoggerFactory}

import scala.collection.JavaConverters._
import scala.util.{Failure, Success, Try}

sealed trait StreamEntry {
  def getName: String

  def getPath: String

  def getInputStream: InputStream

  def isDirectory: Boolean
}

trait XLArchiveEntry extends StreamEntry {
  def path: String

  protected def stream: InputStream

  def ze: ArchiveEntry

  override def getName: String = path.split('/').last

  override def getPath: String = if (isDirectory) {
    path.substring(0, path.length - 1)
  } else {
    path
  }

  override def getInputStream: InputStream = new BufferedInputStream(new InputStream {

    override def read(b: Array[Byte]): Int = stream.read(b)

    override def read(b: Array[Byte], off: Int, len: Int): Int = stream.read(b, off, len)

    override def read(): Int = stream.read()

    override def close(): Unit = {}
  })


}

case class ArchivedEntry(ze: ArchiveEntry, stream: InputStream) extends XLArchiveEntry {
  override val path: String = ze.getName

  override def isDirectory: Boolean = ze.isDirectory
}

case class ZipArchivedEntry(ze: ZipArchiveEntry, stream: InputStream) extends XLArchiveEntry {
  override val path: String = ze.getName

  override def isDirectory: Boolean = ze.isDirectory
}

case class JarArchivedEntry(ze: ZipArchiveEntry, stream: InputStream) extends XLArchiveEntry {
  override val path: String = ze.getName

  override def isDirectory: Boolean = ze.isDirectory
}

case class DirectoryEntry(file: File, base: String) extends StreamEntry {
  override def getName: String = file.getName

  override def getPath: String = file.getPath.substring(base.length)

  override def getInputStream: InputStream = throw new UnsupportedOperationException("directory does not have getInputStream")

  override def isDirectory: Boolean = true
}

case class FileEntry(file: File, base: String) extends StreamEntry {
  override def getName: String = file.getName

  override def getPath: String = file.getPath.substring(base.length)

  override def getInputStream: InputStream = new FileInputStream(file)

  override def isDirectory: Boolean = false
}

case class FileStreamEntry(stream: InputStream, name: String) extends StreamEntry {
  override def getName: String = name

  override def getPath: String = name

  override def getInputStream: InputStream = new InputStream {
    override def read(): Int = stream.read()

    override def read(b: Array[Byte]): Int = stream.read(b)

    override def read(b: Array[Byte], off: Int, len: Int): Int = stream.read(b, off, len)

    override def markSupported(): Boolean = stream.markSupported()

    override def available(): Int = stream.available()

    override def skip(n: Long): Long = stream.skip(n)

    override def reset(): Unit = stream.reset()

    override def close(): Unit = {} // Do not close ourselves, this object does not own the stream

    override def mark(readlimit: Int): Unit = stream.mark(readlimit)
  }

  override def isDirectory = false
}

sealed trait Streamer {
  def stream(): Stream[StreamEntry]
}

object StreamerFactory {
  val logger: Logger = LoggerFactory.getLogger(classOf[StreamerFactory])
  def defaultMappings(): StreamerFactory = forConfig(ConfigFactory.defaultReference())

  def forConfig(config: Config) = new StreamerFactory(new PackagerConfig(config).archiveExtensionMappings)

  type ArchiveStreamProvider = InputStream => ArchiveInputStream
  type ZipArchiveStreamProvider = InputStream => ZipArchiveInputStream
  type JarArchiveStreamProvider = InputStream => JarArchiveInputStream
}

class StreamerFactory(archiveMappings: Map[String, String]) {
  // used for replacement
  private[io] def getArchiveEntryStreamer(file: File): Streamer = {
    val optionalExtMappingTuple: Option[(String, String)] = archiveMappings.find(p = am => file.getName.endsWith('.' + am._1))
    val resettableIs = ArtifactIOUtils.getResettableInputStream(new FileInputStream(file))

    optionalExtMappingTuple match {
      case None => throw UnsupportedArchiveExtensionException(s"${file.getName} is not a supported archive")
      case Some((originExt, mappedExt)) =>
        StreamerFactory.logger.debug(s"Detected mapped archive extension $originExt -> $mappedExt for ${file.getName}")

        if(!archiveTypeMatchesExtension(resettableIs, mappedExt, file.getName)) {
          return new EmptyStreamer()
        }

        mappedExt match {
          case SupportedArchiveExtensions.ZIP => new ZipFileArchiveStreamer(file, is => new ZipArchiveInputStream(is))
          case SupportedArchiveExtensions.JAR => new JarFileArchiveStreamer(file, is => new JarArchiveInputStream(is))
          case SupportedArchiveExtensions.TAR => new ArchiveStreamer(file, is => new TarArchiveInputStream(is))
          case SupportedArchiveExtensions.TARGZ => new ArchiveStreamer(file,
            is => new TarArchiveInputStream(new GzipCompressorInputStream(is)))
          case SupportedArchiveExtensions.TARBZ2 => new ArchiveStreamer(file,
            is => new TarArchiveInputStream(new BZip2CompressorInputStream(is)))
        }
    }
  }

  // used for scanning
  private[io] def getArchiveEntryStreamStreamer(is: InputStream, name: String): Streamer = {
    val optionalExtMappingTuple: Option[(String, String)] = archiveMappings.find(p = am => name.endsWith('.' + am._1))
    val resettableIs = ArtifactIOUtils.getResettableInputStream(is)

    optionalExtMappingTuple match {
      case None => throw UnsupportedArchiveExtensionException(s"$name is not a supported archive")
      case Some((originExt, mappedExt)) =>
        StreamerFactory.logger.debug(s"Detected mapped archive extension $originExt -> $mappedExt for $name")

        if(!archiveTypeMatchesExtension(resettableIs, mappedExt, name)) {
          return new EmptyStreamer()
        }

        mappedExt match {
            case SupportedArchiveExtensions.ZIP => new ZipArchiveStreamStreamer(resettableIs, is => new ZipArchiveInputStream(is))
            case SupportedArchiveExtensions.JAR => new JarArchiveStreamStreamer(resettableIs, is => new JarArchiveInputStream(is))
            case SupportedArchiveExtensions.TAR => new ArchiveStreamStreamer(resettableIs, is => new TarArchiveInputStream(is))
            case SupportedArchiveExtensions.TARGZ => new ArchiveStreamStreamer(resettableIs,
              is => new TarArchiveInputStream(new GzipCompressorInputStream(is)))
            case SupportedArchiveExtensions.TARBZ2 => new ArchiveStreamStreamer(resettableIs,
              is => new TarArchiveInputStream(new BZip2CompressorInputStream(is)))
          }
    }
  }

  def outputStream(target: File): ArchiveOutputStream = {
      val name = target.getName
      archiveMappings.find(p = am => name.endsWith('.' + am._1)) match {
      case None => throw UnsupportedArchiveExtensionException(s"$name is not a supported archive")
      case Some((originExt, mappedExt)) =>
        StreamerFactory.logger.debug(s"Detected mapped archive extension $originExt -> $mappedExt for $name")
        mappedExt match {
          case SupportedArchiveExtensions.ZIP => new ZipArchiveOutputStream(target)
          case SupportedArchiveExtensions.JAR => new ZipArchiveOutputStream(target) // JarArchiveOutputStream doesn't have this constructor. It is required for correct function.
          case SupportedArchiveExtensions.TAR => configureTarArchiveOutputStream(
            new TarArchiveOutputStream(new FileOutputStream(target)))
          case SupportedArchiveExtensions.TARGZ => configureTarArchiveOutputStream(
            new TarArchiveOutputStream(new GzipCompressorOutputStream(new FileOutputStream(target))))
          case SupportedArchiveExtensions.TARBZ2 => configureTarArchiveOutputStream(
            new TarArchiveOutputStream(new BZip2CompressorOutputStream(new FileOutputStream(target))))
          case _ => throw UnsupportedArchiveExtensionException(
            s"$name with extension $originExt -> $mappedExt is not a supported archive")
        }
    }
  }

  def configureTarArchiveOutputStream(os: TarArchiveOutputStream): TarArchiveOutputStream = {
    os.setLongFileMode(TarArchiveOutputStream.LONGFILE_POSIX)
    os.setBigNumberMode(TarArchiveOutputStream.BIGNUMBER_POSIX)
    os
  }

  def archiveTypeMatchesExtension(is: InputStream, expectedType: String, name: String): Boolean = {
    val detectedArchiveType: Option[String] = Try(ArchiveStreamFactory.detect(is)) match {
      case Success(archiveType) => Option(archiveType)
      case Failure(_) => Try(CompressorStreamFactory.detect(is)) match {
        case Success(archiveType) => Option(archiveType)
        case Failure(_) => Option.empty[String]
      }
    }

    (detectedArchiveType, expectedType) match {
      case (Some(CompressorStreamFactory.BZIP2), SupportedArchiveExtensions.TARBZ2) => true
      case (Some(CompressorStreamFactory.GZIP), SupportedArchiveExtensions.TARGZ) => true
      case (Some(ArchiveStreamFactory.TAR), SupportedArchiveExtensions.TAR) => true
      case (Some(ArchiveStreamFactory.ZIP), SupportedArchiveExtensions.ZIP | SupportedArchiveExtensions.JAR) => true
      case (Some(detected), _) =>
        StreamerFactory.logger.warn(
          s"Found archive ($name) with mapped type .$expectedType, but the actual type is .$detected - skipping")
        false
      case (
        None,
        SupportedArchiveExtensions.TARBZ2 |
        SupportedArchiveExtensions.TARGZ |
        SupportedArchiveExtensions.TAR |
        SupportedArchiveExtensions.ZIP |
        SupportedArchiveExtensions.JAR) =>
        StreamerFactory.logger.warn(
          s"Found file ($name) with mapped type .$expectedType, but the file is not a valid archive type")
        false
      case (None, _) =>
        throw UnsupportedArchiveExtensionException(s"$name with extension $expectedType is not a supported archive")
    }
  }

  def hasArchiveExtension(name: String): Boolean = {
    val optionalExtMappingTuple: Option[(String, String)] = archiveMappings.find(p = am => name.endsWith('.' + am._1))
    optionalExtMappingTuple.isDefined
  }

  def streamer(is: InputStream, name: String): Streamer = {
    if (hasArchiveExtension(name)) {
      getArchiveEntryStreamStreamer(is, name)
    } else {
      new FileStreamStreamer(is, name)
    }
  }

  def streamer(file: File): Streamer = {
    if (file.isDirectory) {
      new DirectoryStreamer(file, this)
    } else if (hasArchiveExtension(file.getName)) {
      getArchiveEntryStreamer(file)
    } else {
      new FileStreamer(file)
    }
  }
}

class DirectoryStreamer(dir: File, sf: StreamerFactory) extends Streamer {
  override def stream(): Stream[StreamEntry] = {
    def fileStream(dir: File, base: String): Stream[StreamEntry] =
      if (dir.isDirectory)
        Option(dir.listFiles)
          .map(a => JArrays.asList(a:_*).asScala.sortBy(_.getPath).toStream.flatMap(file =>
            if (file.isDirectory) {
              DirectoryEntry(file, base) #:: fileStream(file, base)
            } else {
              FileEntry(file, base) #:: fileStream(file, base)
            }))
          .getOrElse(Stream.empty)
      else Stream.empty

    val base = dir.getPath + "/"
    fileStream(dir, base)
  }
}

class EmptyStreamer() extends Streamer {
  override def stream(): Stream[StreamEntry] = Stream.empty[StreamEntry]
}

class FileStreamer(file: File) extends Streamer {
  override def stream(): Stream[StreamEntry] = FileEntry(file, file.getParentFile.getPath + "/") #:: Stream.empty[StreamEntry]
}

class FileStreamStreamer(is: InputStream, name: String) extends Streamer {
  override def stream(): Stream[StreamEntry] = FileStreamEntry(is, name) #:: Stream.empty[StreamEntry]
}

trait ArchiveEntryStreamer extends Streamer {
  def nextEntry(s: ArchiveInputStream, toClose: Set[Closeable]): Stream[StreamEntry] = {
    Option(s.getNextEntry) match {
      case None =>
        toClose.foreach(c => Try(c.close()))
        Stream.empty
      case Some(e) => ArchivedEntry(e, s) #:: nextEntry(s, toClose)
    }
  }
}

trait ZipEntryStreamer extends Streamer {
  def nextEntry(s: ZipArchiveInputStream, toClose: Set[Closeable]): Stream[StreamEntry] = {
    Option(s.getNextZipEntry) match {
      case None =>
        toClose.foreach(c => Try(c.close()))
        Stream.Empty
      case Some(e) => ZipArchivedEntry(e, s) #:: nextEntry(s, toClose)
    }
  }
}

trait ZipFileEntryStreamer extends Streamer {
  def nextEntry(zf: ZipFile, zfe: JEnumeration[ZipArchiveEntry], toClose: Set[Closeable]): Stream[StreamEntry] = {
    if (zfe.hasMoreElements) {
      val e = zfe.nextElement()
      val eStream = zf.getInputStream(e)
      ZipArchivedEntry(e, eStream) #:: nextEntry(zf, zfe, toClose + eStream)
    } else {
      toClose.foreach(c => Try(c.close()))
      Stream.Empty
    }
  }
}

trait JarEntryStreamer extends Streamer {
  def nextEntry(s: JarArchiveInputStream, toClose: Set[Closeable]): Stream[StreamEntry] = {
    Option(s.getNextJarEntry) match {
      case None =>
        toClose.foreach(c => Try(c.close()))
        Stream.Empty
      case Some(e) => JarArchivedEntry(e, s) #:: nextEntry(s, toClose)
    }
  }
}

trait JarFileEntryStreamer extends Streamer {
  def nextEntry(zf: ZipFile, zfe: JEnumeration[ZipArchiveEntry], toClose: Set[Closeable]): Stream[StreamEntry] = {
    if (zfe.hasMoreElements) {
      val e = zfe.nextElement()
      val eStream = zf.getInputStream(e)
      JarArchivedEntry(e, eStream) #:: nextEntry(zf, zfe, toClose + eStream)
    } else {
      toClose.foreach(c => Try(c.close()))
      Stream.Empty
    }
  }
}

class ZipFileArchiveStreamer(file: File, streamProvider: ZipArchiveStreamProvider) extends ZipFileEntryStreamer {
  override def stream(): Stream[StreamEntry] = {
    val is = new FileInputStream(file)
    val jdkIs = streamProvider(is)
    val zf = new ZipFile(file)
    nextEntry(zf, zf.getEntriesInPhysicalOrder, Set(is, jdkIs))
  }
}

class ZipArchiveStreamStreamer(is: InputStream, streamProvider: ZipArchiveStreamProvider) extends ZipEntryStreamer {
  override def stream(): Stream[StreamEntry] = {
    val jdkIs = streamProvider(is)
    nextEntry(jdkIs, Set())
  }
}

class JarFileArchiveStreamer(file: File, streamProvider: JarArchiveStreamProvider) extends JarFileEntryStreamer {
  override def stream(): Stream[StreamEntry] = {
    val is = new FileInputStream(file)
    val jdkIs = streamProvider(is)
    val zf = new ZipFile(file)
    nextEntry(zf, zf.getEntriesInPhysicalOrder, Set(is, jdkIs))
  }
}

class JarArchiveStreamStreamer(is: InputStream, streamProvider: JarArchiveStreamProvider) extends JarEntryStreamer {
  override def stream(): Stream[StreamEntry] = {
    val jdkIs = streamProvider(is)
    nextEntry(jdkIs, Set())
  }
}

class ArchiveStreamer(file: File, val streamProvider: ArchiveStreamProvider) extends ArchiveEntryStreamer {
  override def stream(): Stream[StreamEntry] = {
    val is = new FileInputStream(file)
    val archiveStream = streamProvider(is)
    nextEntry(archiveStream, Set(is, archiveStream))
  }
}

class ArchiveStreamStreamer(is: InputStream, val streamProvider: ArchiveStreamProvider) extends ArchiveEntryStreamer {
  override def stream(): Stream[StreamEntry] = {
    val archiveStream = streamProvider(is)
    nextEntry(archiveStream, Set())
  }
}

case class UnsupportedArchiveExtensionException(str: String) extends RuntimeException(str)
