/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager

import java.io._
import java.nio.charset.{Charset, StandardCharsets}
import java.security.{DigestInputStream, MessageDigest}
import java.util.Calendar
import java.util.regex.Pattern
import java.util.regex.Pattern.{CASE_INSENSITIVE, COMMENTS}
import java.util.zip.CRC32

import com.xebialabs.deployit.plugin.api.udm.artifact.{DerivedArtifact, FolderArtifact, SourceArtifact}
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployableArtifact.SCAN_PLACEHOLDERS_PROPERTY_NAME
import com.xebialabs.deployit.util.{BOM, DetectBOM, TryWith}
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.xldeploy.packager.SourceArtifactEnricher._
import com.xebialabs.xldeploy.packager.io._
import org.apache.commons.codec.binary.Hex
import org.apache.commons.compress.archivers.jar.JarArchiveEntry
import org.apache.commons.compress.archivers.tar.{TarArchiveEntry, TarArchiveInputStream}
import org.apache.commons.compress.archivers.zip.{ZipArchiveEntry, ZipMethod}
import org.apache.commons.compress.archivers.{ArchiveEntry, ArchiveOutputStream}
import org.apache.commons.io.IOUtils
import org.slf4j.{Logger, LoggerFactory}

import scala.collection.JavaConverters._
import scala.language.postfixOps
import scala.util.{Failure, Success, Try}

object SourceArtifactEnricher {
  private[SourceArtifactEnricher] val logger: Logger = LoggerFactory.getLogger(classOf[SourceArtifactEnricher])

  private[SourceArtifactEnricher] sealed trait ScanType

  private[SourceArtifactEnricher] case object DigestOnly extends ScanType

  private[SourceArtifactEnricher] case object ProcessArchive extends ScanType

  private[SourceArtifactEnricher] case object ProcessTextFile extends ScanType

  private[SourceArtifactEnricher] case class ScanSpec(scanPlaceholders: Boolean, excludePathRegex: Option[Pattern], textFileRegex: Pattern) {
    def this(fa: SourceArtifact) = this(
      shouldScanPlaceholders(fa),
      Option(fa.getExcludeFileNamesRegex).filterNot(_.isEmpty).map(compilePattern),
      compilePattern(fa.getTextFileNamesRegex)
    )
  }

  implicit class ScanHandler(scanSpec: ScanSpec)(implicit streamerFactory: StreamerFactory) {
    def getScanType(entry: StreamEntry): ScanType = {
      if (!scanSpec.scanPlaceholders) DigestOnly
      else if (scanSpec.excludePathRegex.exists(_.matcher(entry.getPath).matches())) DigestOnly
      else if (streamerFactory.isArchive(entry.getName)) ProcessArchive
      else if (scanSpec.textFileRegex.matcher(entry.getName).matches()) ProcessTextFile
      else DigestOnly
    }
  }

  val defaultDelims = "{{ }}"
  var patternMap: Map[String, Pattern] = Map()
  val devNull = new DevNull
  val devNullWriter = new OutputStreamWriter(devNull)

  private[this] def compilePattern(regex: String): Pattern = patternMap.get(regex) match {
    case Some(compiled) => compiled
    case None =>
      val compiled = Pattern.compile(regex, COMMENTS | CASE_INSENSITIVE)
      patternMap += (regex -> compiled)
      compiled
  }

  private[SourceArtifactEnricher] def hasCheckSum(bda: SourceArtifact): Boolean = {
    bda.getChecksum != null && !bda.getChecksum.isEmpty
  }

  private[SourceArtifactEnricher] def getDelimiters(artifact: SourceArtifact): String = if (artifact.hasProperty("delimiters")) {
    artifact.getProperty[String]("delimiters")
  } else {
    defaultDelims
  }


  private[SourceArtifactEnricher] def shouldScanPlaceholders(artifact: SourceArtifact): Boolean = {
    !artifact.hasProperty(SCAN_PLACEHOLDERS_PROPERTY_NAME) || artifact.getProperty[Boolean](SCAN_PLACEHOLDERS_PROPERTY_NAME)
  }

  private[SourceArtifactEnricher] def fullArtifactPath(entry: StreamEntry, sa: SourceArtifact): String = {
    if (sa.getFile.isDirectory) {
      sa.getFile.getPath + File.separator + entry.getPath
    } else {
      sa.getFile.getPath
    }
  }
}

case class StreamFile(name: String, inputStream: InputStream)

class SourceArtifactEnricher(streamerFactory: StreamerFactory) {
  implicit val sf: StreamerFactory = streamerFactory

  def enrichArtifact(sa: SourceArtifact, messageDigest: () => MessageDigest): Unit = {
    enrichArtifact(sa, None, devNull, messageDigest)
  }

  def enrichArtifact(sa: SourceArtifact, streamFile: StreamFile, messageDigest: () => MessageDigest): Unit = {
    enrichArtifact(sa, Option(streamFile), devNull, messageDigest)
  }

  def enrichArtifact(sa: SourceArtifact, streamFile: Option[StreamFile], os: OutputStream, messageDigest: () => MessageDigest): Unit = {
      val digest: Option[MessageDigest] = if (!hasCheckSum(sa)) Some(messageDigest()) else None

      if (shouldScanPlaceholders(sa) || !hasCheckSum(sa)) {
        val mustacher = Mustacher(getDelimiters(sa))

        sa match {
          case fa: FolderArtifact =>
            logger.info(s"Going to scan artifact $fa as a folder artifact")
            scanFolderArtifact(fa, streamFile, digest, mustacher)
          case _ if streamFile.isEmpty =>
            logger.info(s"Going to scan artifact $sa")
            TryWith(sa.getFile.getInputStream) { fis =>
              scanFileOrArchiveArtifact(sa, StreamFile(sa.getFile.getName, fis), os, digest, mustacher)
            }
          case _ =>
            logger.info(s"Going to scan artifact $sa from stream")
            scanFileOrArchiveArtifact(sa, streamFile.get, os, digest, mustacher)
        }

        sa.setPlaceholders(mustacher.placeholders.asJava)
        digest.foreach(d => sa.setProperty(SourceArtifact.CHECKSUM_PROPERTY_NAME, Hex.encodeHexString(d.digest())))
      } else {
        logger.info(s"Not enriching $sa as no placeholders should be scanned and checksum is present")
      }
  }

  def enrichArtifact(da: DerivedArtifact[_ <: SourceArtifact]): Unit = {
      if (da.getSourceArtifact == null) {
        da.setFile(null)
      } else {
        val derivedFile = createDerivedFile(da)
        da.setFile(derivedFile)
      }
  }

  def createDerivedFile(da: DerivedArtifact[_ <: SourceArtifact]): OverthereFile = {
    val sa = da.getSourceArtifact
    val sourceFile = da.getSourceArtifact.getFile
    val workDir = sourceFile.getParentFile
    val derivedFile = OverthereUtils.getUniqueFolder(workDir, sa.getName).getFile(sourceFile.getName)
    sourceFile.copyTo(derivedFile)

    replacePlaceholdersInArtifact(da, derivedFile)
    derivedFile
  }

  private[this] def updateDigestWithFilename(entry: StreamEntry, digest: MessageDigest): Unit = {
    var digestPath = entry.getPath + (if (entry.isDirectory) File.separator else "")
    digestPath = digestPath.replace("\\", "/")
    logger.trace(s"scanFolderArtifact: Digest path [$digestPath]")
    digest.update(digestPath.getBytes(StandardCharsets.UTF_8))
  }

  private[this] def scanFolderArtifact(fa: FolderArtifact, streamFile: Option[StreamFile], digest: Option[MessageDigest],
                                       mustacher: Mustacher): Unit = {
    val streamer = streamFile match {
      case Some(x) => streamerFactory.streamer(x.inputStream, x.name)
      case None => streamerFactory.streamer(fa.getFile)
    }
    streamer.stream().foreach { entry =>
      digest.foreach(d => updateDigestWithFilename(entry, d))
      if (!entry.isDirectory) {
        doScan(fa, entry, mustacher, is => digest.map(new DigestInputStream(is, _)).getOrElse(is))
      }
    }
  }

  private[this] def scanFileOrArchiveArtifact(sa: SourceArtifact, streamFile: StreamFile, os: OutputStream,
                                              digest: Option[MessageDigest], mustacher: Mustacher): Unit = {
    val inputStream = digest.map(new DigestInputStream(streamFile.inputStream, _)).getOrElse(streamFile.inputStream)
    if (shouldScanPlaceholders(sa)) {
      streamerFactory.streamer(inputStream, streamFile.name).stream().foreach { entry =>
        if (!entry.isDirectory) {
          doScan(sa, entry, mustacher, identity[InputStream])
        }
      }
      // Ensure that we empty the stream so that all bytes are read, testcases seem to prove this is not needed,
      // But we don't want to accidentally calculate wrong checksum.
      OverthereUtils.write(inputStream, os)
    } else {
      logger.debug(s"Artifact [$sa] has disabled placeholder scanning")
      OverthereUtils.write(inputStream, os)
    }
  }

  private[this] def doScan(sa: SourceArtifact, entry: StreamEntry, mustacher: Mustacher,
                           transform: InputStream => InputStream): Try[Unit] = {

    val scanSpec = new ScanSpec(sa)
    TryWith(entry.getInputStream) { inputStream =>
      val is = transform(inputStream)

      scanSpec.getScanType(entry) match {
        case DigestOnly =>
          logger.debug(s"$sa: Skipping ${entry.getName}")
          OverthereUtils.write(is, devNull) // Just write the bytes

        case ProcessArchive =>
          logger.debug(s"$sa: Detected archive for ${entry.getName}")
          streamerFactory.streamer(is, entry.getName).stream().foreach { e2 =>
            if (!e2.isDirectory) {
              doScan(sa, e2, mustacher, identity[InputStream])
            }
          }
          // Ensure that we empty the stream so that all bytes are read, so that we don't accidentally calculate wrong checksum.
          OverthereUtils.write(is, devNull) // Finish the archive

        case ProcessTextFile =>
          logger.debug(s"$sa: ${entry.getName} is a text file.")
          val resettableInputStream = getResettableInputStream(is)
          val reader: InputStreamReader = DetectBOM.detect(resettableInputStream) match {
            case BOM.NONE => new InputStreamReader(resettableInputStream)
            case bom: BOM => new InputStreamReader(resettableInputStream, bom.getCharset)
          }
          TryWith(mustacher.newReader(reader)) { r =>
            OverthereUtils.write(r, devNullWriter) // Scan the file for placeholders using the Mustacher
          }
      }
    }
  }

  private def getResettableInputStream(is: InputStream): InputStream = {
    Try(is.reset()) match {
      case Failure(_) => new BufferedInputStream(is)
      case Success(_) if is.isInstanceOf[TarArchiveInputStream] => new BufferedInputStream(is) // it does not reset correctly
      case _ => is
    }
  }

  private[this] def replacePlaceholdersInArtifact(da: DerivedArtifact[_ <: SourceArtifact], derivedFile: OverthereFile): Unit = {
    val sa = da.getSourceArtifact
    if (shouldScanPlaceholders(sa)) {
      val mustacher = MustacherReplacer(getDelimiters(sa))
      val scanSpec = new ScanSpec(sa)

      sa match {
        case fa: FolderArtifact =>
          logger.info(s"Going to replace placeholders in artifact $fa as in a folder artifact")
          replacePlaceholdersFolderArtifact(da, derivedFile, mustacher, scanSpec)

        case _ if streamerFactory.isArchive(sa.getFile.getName) =>
          logger.info(s"Going to replace placeholders in artifact $sa as in an archive artifact")
          replacePlaceholdersArchiveArtifact(da, derivedFile, mustacher, scanSpec)

        case _ =>
          logger.info(s"Going to replace placeholders in artifact $sa from stream")
          replacePlaceholdersFileArtifact(da, derivedFile, mustacher, scanSpec)

      }
    } else {
      logger.info(s"Not enriching $sa as no placeholders should be scanned")
    }
  }

  private[this] def replacePlaceholdersFolderArtifact(sa: DerivedArtifact[_ <: SourceArtifact], derivedFile: OverthereFile,
                                                      mustacher: MustacherReplacer, scanSpec: ScanSpec): Unit = {
    val streamer = streamerFactory.streamer(sa.getSourceArtifact.getFile)
    streamer.stream().foreach { entry =>
      doReplacePlaceholders(sa, derivedFile, derivedFile.getPath, null, entry, mustacher, scanSpec)
    }
  }

  private[this] def replacePlaceholdersFileArtifact(sa: DerivedArtifact[_ <: SourceArtifact], derivedFile: OverthereFile,
                                                    mustacher: MustacherReplacer, scanSpec: ScanSpec): Unit = {
    val streamer = streamerFactory.streamer(sa.getSourceArtifact.getFile)
    streamer.stream().foreach { entry =>
      doReplacePlaceholders(sa, derivedFile, derivedFile.getPath, null, entry, mustacher, scanSpec)
    }
  }

  private[this] def replacePlaceholdersArchiveArtifact(da: DerivedArtifact[_ <: SourceArtifact], derivedFile: OverthereFile,
                                                       mustacher: MustacherReplacer, scanSpec: ScanSpec): Unit = {
    val path = derivedFile.getPath
    withArchiveOutputStream(path, os => {
      val sa = da.getSourceArtifact
      streamerFactory.streamer(sa.getFile).stream().foreach {
        case e: XLArchiveEntry => doReplacePlaceholdersArchivedEntry(os, e.getInputStream, derivedFile.getParentFile.getPath, e, sa, mustacher, da.getPlaceholders.asScala.toMap, scanSpec)
      }
    })
  }

  private[this] def doReplacePlaceholders(da: DerivedArtifact[_ <: SourceArtifact], derivedFile: OverthereFile, basePath: String, os: OutputStreamWriter,
                                          entry: StreamEntry, mustacher: MustacherReplacer, scanSpec: ScanSpec): Try[Unit] = {
    TryWith(entry.getInputStream) { is =>
      val sa = da.getSourceArtifact
      scanSpec.getScanType(entry) match {
        case DigestOnly =>
          logger.debug(s"$da: Skipping ${entry.getName}")
          val targetPath = derivedFile.getPath
          val os = new FileOutputStream(targetPath)
          OverthereUtils.write(is, os) // Just write the bytes

        case ProcessArchive if sa.isInstanceOf[FolderArtifact] =>
          logger.debug(s"$da: Detected archive for ${entry.getName} in Folder artifact")
          val path = derivedFile.getPath + File.separator + entry.getPath
          withArchiveOutputStream(path, os => {
            streamerFactory.streamer(entry.getInputStream, entry.getName).stream().foreach {
              case e: XLArchiveEntry => doReplacePlaceholdersArchivedEntry(os, e.getInputStream, basePath, e, sa, mustacher, da.getPlaceholders.asScala.toMap, scanSpec)
            }
          })

        case ProcessArchive =>
          logger.debug(s"$da: Detected archive for ${entry.getName}")
          val path = derivedFile.getPath
          withArchiveOutputStream(path, os => {
            doReplacePlaceholdersArchivedEntry(os, is, basePath, entry.asInstanceOf[XLArchiveEntry], sa, mustacher, da.getPlaceholders.asScala.toMap, scanSpec)
          })

        case ProcessTextFile =>
          logger.debug(s"$da: ${entry.getName} is a text file.")
          doReplacePlaceholdersTextFile(da, derivedFile, basePath, os, is, entry, mustacher)
      }
    }
  }

  private[this] def doReplacePlaceholdersTextFile(sa: DerivedArtifact[_ <: SourceArtifact], derivedFile: OverthereFile, basePath: String,
                                                  os: OutputStreamWriter, is: InputStream, entry: StreamEntry, mustacher: MustacherReplacer): Unit = {
    val resettableInputStream = getResettableInputStream(is)

    val charset = DetectBOM.detect(resettableInputStream) match {
      case BOM.NONE => getCharset(sa.getSourceArtifact, entry)
      case bom: BOM => Option(bom.getCharset)
    }

    val reader = charset.map(cs => new InputStreamReader(resettableInputStream, cs)).getOrElse(new InputStreamReader(resettableInputStream))

    val wrappedWriter = if (Option(os).isEmpty) {
      val targetPath = if (derivedFile.isDirectory) {
        basePath + File.separator + entry.getPath
      } else {
        basePath
      }
      charset.map(cs => new OutputStreamWriter(new FileOutputStream(targetPath), cs.newEncoder())).getOrElse(new OutputStreamWriter(new FileOutputStream(targetPath)))
    } else {
      os
    }

    TryWith(mustacher.newReader(reader, sa.getPlaceholders.asScala.toMap)) { r =>
      TryWith(wrappedWriter) { w =>
        OverthereUtils.write(r, w)
      }
    }
  }

  private[this] def doReplacePlaceholdersArchivedEntry(os: ArchiveOutputStream, is: InputStream, basePath: String, entry: XLArchiveEntry, sa: SourceArtifact,
                                                       mustacher: MustacherReplacer, placeholders: Map[String, String], scanSpec: ScanSpec): Unit = {
    logger.debug(s"$sa: ${entry.getName} is a archive entry file.")
    processEntry(is, os, basePath, entry, mustacher, placeholders, scanSpec, sa)
  }

  private[this] def processEntry(is: InputStream, os: ArchiveOutputStream, basePath: String, entry: XLArchiveEntry, mustacher: MustacherReplacer,
                                 placeholders: Map[String, String], scanSpec: ScanSpec, sa: SourceArtifact): Unit = {
    val resettableInputStream = getResettableInputStream(is)

    scanSpec.getScanType(entry) match {
      case ProcessArchive =>
        processArchive()
      case ProcessTextFile if !entry.isDirectory =>
        processTextFile()
      case _ =>
        processOtherFileOrFolder()
    }

    def processArchive() : Unit = {
      logger.debug(s"Replacing placeholders in ${entry.getName} for path ${entry.getPath} as in an archive artifact")
      val file = new File(basePath + File.separator + Calendar.getInstance().getTimeInMillis + entry.getName)
      logger.debug(s"Creating a temporary file for ${entry.getName} in path ${file.getAbsolutePath}")
      file.createNewFile()
      withArchiveOutputStream(file.getAbsolutePath, temporaryArchiveOutputStream => {
        streamerFactory.streamer(entry.getInputStream, entry.getName).stream().foreach {
          case inner: XLArchiveEntry if inner != entry =>
            logger.debug(s"Processing entry [${inner.getName}] with path [${inner.getPath}] for parent entry [${entry.getName}]")
            processEntry(inner.getInputStream, temporaryArchiveOutputStream, basePath, inner, mustacher, placeholders, scanSpec, sa)
        }
      })
      logger.debug(s"Processed entry [${entry.getName}]")
      val tempFIS = new FileInputStream(file)
      TryWith(tempFIS) { archIs =>
        val (ze, convertedInputStream) = entry match {
          case e: JarArchivedEntry =>
            val (updated, buf) = recalculateSizeAndCrc(copyExtraJarEntryData(e.ze), IOUtils.toByteArray(archIs))
            (updated, new ByteArrayInputStream(buf))
          case e: ZipArchivedEntry =>
            val (updated, buf) = recalculateSizeAndCrc(copyExtraZipEntryData(e.ze), IOUtils.toByteArray(archIs))
            (updated, new ByteArrayInputStream(buf))
          case e: ArchivedEntry =>
            val tae = new TarArchiveEntry(e.ze.getName)
            val buf = IOUtils.toByteArray(archIs)
            tae.setSize(buf.size)
            (tae, new ByteArrayInputStream(buf))
          case e =>
            (e.ze, archIs)
        }
        withOpenArchiveEntry(os, ze, () => {
          OverthereUtils.write(convertedInputStream, os)
        })
      } match {
        case Success(_) =>
          logger.debug(s"Copied data to the parent archive for entry [${entry.getName}]")
          file.delete()
          logger.debug(s"Deleted temporary data")
        case Failure(err) =>
          logger.error(s"Could not copy data from temporary file")
          logger.error(err.getMessage, err)
      }
    }

    def processTextFile(): Unit = {
      logger.debug(s"Replacing placeholders in ${entry.getName} for path ${entry.getPath}")
      val charset = DetectBOM.detect(resettableInputStream) match {
        case BOM.NONE => getCharset(sa, entry)
        case bom: BOM => Option(bom.getCharset)
      }

      logger.debug(s"Entry ${entry.getName} has encoding: ${charset.getOrElse("Unknown")}")

      val reader = charset.map(cs => new InputStreamReader(resettableInputStream, cs)).getOrElse(new InputStreamReader(resettableInputStream))

      val (ze, convertedInputStream, processed) = entry match {
        case e: JarArchivedEntry if e.method == ZipMethod.STORED =>
          val wrappedReader = mustacher.newReader(reader, placeholders)
          val (updated, buf) = recalculateSizeAndCrc(copyExtraJarEntryData(e.ze), wrappedReader, charset)
          (updated, new ByteArrayInputStream(buf), true)
        case e: JarArchivedEntry =>
          (copyExtraJarEntryData(e.ze), resettableInputStream, false)
        case e: ZipArchivedEntry if e.method == ZipMethod.STORED =>
          val wrappedReader = mustacher.newReader(reader, placeholders)
          val (updated, buf) = recalculateSizeAndCrc(copyExtraZipEntryData(e.ze), wrappedReader, charset)
          (updated, new ByteArrayInputStream(buf), true)
        case e: ZipArchivedEntry =>
          (copyExtraZipEntryData(e.ze), resettableInputStream, false)
        case e: ArchivedEntry =>
          val wrappedReader = mustacher.newReader(reader, placeholders)
          val tae = new TarArchiveEntry(e.ze.getName)
          val buf = charset.map(cs => IOUtils.toByteArray(wrappedReader, cs)).getOrElse(IOUtils.toByteArray(wrappedReader, Charset.defaultCharset()))
          tae.setSize(buf.size)
          (tae, new ByteArrayInputStream(buf), true)
        case e => (e.ze, resettableInputStream, false)
      }

      withOpenArchiveEntry(os, ze, () => {
        val convertedReader = charset.map(cs => new InputStreamReader(convertedInputStream, cs)).getOrElse(new InputStreamReader(convertedInputStream))
        val wrappedReader = if (processed) {
          convertedReader
        } else {
          mustacher.newReader(convertedReader, placeholders)
        }
        copyBytes(wrappedReader, os, charset)
      })
    }

    def processOtherFileOrFolder(): Unit = {  // other files and folders, we cannot skip folders because those can be used
      logger.debug(s"Processing ${entry.getName} for path ${entry.getPath}")
      val ze = entry match {
        case e: JarArchivedEntry => copyExtraJarEntryData(e.ze)
        case e: ZipArchivedEntry => copyExtraZipEntryData(e.ze)
        case e: ArchivedEntry => e.ze
      }
      withOpenArchiveEntry(os, ze, () => {
        OverthereUtils.write(resettableInputStream, os)
      })
    }
  }

  private[this] def copyExtraZipEntryData(entry: ZipArchiveEntry): ZipArchiveEntry = {
    copyExtraEntryData[ZipArchiveEntry](entry, entry => new ZipArchiveEntry(entry))
  }

  private[this] def copyExtraJarEntryData(entry: JarArchiveEntry): JarArchiveEntry = {
    copyExtraEntryData[JarArchiveEntry](entry, entry => new JarArchiveEntry(entry))
  }

  private[this] def copyExtraEntryData[T <: ZipArchiveEntry](entry: T, wrapper: T => T): T = {
    val cde = entry.getCentralDirectoryExtra
    val extra = entry.getExtra
    val version = entry.getVersionMadeBy
    val versionReq = entry.getVersionRequired
    val wrapped = wrapper(entry)
    if (entry.getPlatform == ZipArchiveEntry.PLATFORM_UNIX) wrapped.setUnixMode(entry.getUnixMode)
    wrapped.setCentralDirectoryExtra(cde)
    wrapped.setExtra(extra)
    wrapped.setVersionMadeBy(version)
    wrapped.setVersionRequired(versionReq)
    wrapped
  }

  private[this] def recalculateSizeAndCrc[T <: ZipArchiveEntry](entry: T, reader: Reader, maybeCharset: Option[Charset]): (T, Array[Byte]) = {
    val buf = maybeCharset.map(cs => IOUtils.toByteArray(reader, cs)).getOrElse(IOUtils.toByteArray(reader, Charset.defaultCharset()))
    entry.setSize(buf.length)
    val crc = new CRC32
    crc.update(buf)
    entry.setCrc(crc.getValue)
    (entry, buf)
  }

  private[this] def recalculateSizeAndCrc[T <: ZipArchiveEntry](entry: T, buf: Array[Byte]): (T, Array[Byte]) = {
    entry.setSize(buf.length)
    val crc = new CRC32
    crc.update(buf)
    entry.setCrc(crc.getValue)
    (entry, buf)
  }

  private[this] def withOpenArchiveEntry(os: ArchiveOutputStream, ae: ArchiveEntry, process: () => Unit): Unit = {
    os.putArchiveEntry(ae)
    process()
    os.closeArchiveEntry()
  }

  private[this] def withArchiveOutputStream(path: String, block: ArchiveOutputStream => Unit): Unit = {
    val os = streamerFactory.outputStream(new FileOutputStream(path), path)
    block(os)
    os.finish()
    os.close()
  }

  private[this] def copyBytes(wrappedReader: Reader, os: OutputStream, maybeCharset: Option[Charset]): Unit = {
    val bytes = maybeCharset.map(cs => IOUtils.toByteArray(wrappedReader, cs)).getOrElse(IOUtils.toByteArray(wrappedReader, Charset.defaultCharset()))
    os.write(bytes)
  }

  private[this] def getCharset(artifact: SourceArtifact, file: StreamEntry): Option[Charset] = {
    val path = fullArtifactPath(file, artifact)
    val fileEncodings = artifact.getFileEncodings
    for (regexToCharset <- fileEncodings.entrySet.asScala) {
      if (path.matches(regexToCharset.getKey)) {
        logger.debug("Relative path [{}] matched regex [{}], using charset [{}]", path, regexToCharset.getKey, regexToCharset.getValue)
        return Option(Charset.forName(regexToCharset.getValue))
      }
    }
    None
  }
}
