package com.xebialabs.deployit.packager;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.io.ByteStreams;
import com.google.common.io.Closeables;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Version;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.FolderArtifact;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.RuntimeIOException;

import de.schlichtherle.truezip.file.TArchiveDetector;
import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.fs.FsSyncException;

import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static com.xebialabs.deployit.packager.CiNameCache.id;
import static com.xebialabs.deployit.packager.truezip.TFiles.newOutputSupplier;
import static java.lang.String.format;
import static java.util.EnumSet.of;

public class DarPackager {

    private final CiNameCache cache;
    private ManifestWriterDriver manifestDriver;

    public DarPackager(ManifestWriter writer) {
        cache = new CiNameCache();
        manifestDriver = new ManifestWriterDriver(cache, writer);
    }

    public TFile buildPackage(Version version, String targetDir) {
        String darName = darName(version);
        TFile darPackage = new TFile(targetDir, darName);
        try {
            manifestDriver.writeToPackage(version, darPackage);
            writeArtifactsToPackage(version, darPackage);
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } finally {
            try {
                TFile.umount(darPackage);
            } catch (FsSyncException e) {
                logger.warn("Couldn't cleanly umount the dar package {}", darPackage);
            }
        }
        return darPackage;
    }

    private void writeArtifactsToPackage(ConfigurationItem ci, TFile darPackage) {
        if (ci instanceof FolderArtifact) {
            OverthereFile sourceFile = ((Artifact) ci).getFile();
            logger.debug("Writing {} to package {}", sourceFile, darPackage);
            TFile targetFolder = new TFile(darPackage, id(cache.lookup(ci), sourceFile.getName()));
            targetFolder.mkdirs();
            copyFolder(sourceFile, targetFolder);
        } else if (ci instanceof Artifact) {
            OverthereFile sourceFile = ((Artifact) ci).getFile();
            logger.debug("Writing {} to package {}", sourceFile, darPackage);
            TFile targetFile = new TFile(darPackage, id(cache.lookup(ci), sourceFile.getName()), TArchiveDetector.NULL);
            copyFile(sourceFile, targetFile);
        }
        for (PropertyDescriptor pd : ci.getType().getDescriptor().getPropertyDescriptors()) {
            if (pd.isAsContainment() && of(LIST_OF_CI, SET_OF_CI).contains(pd.getKind())) {
                for (ConfigurationItem c : (Collection<ConfigurationItem>) pd.get(ci)) {
                    writeArtifactsToPackage(c, darPackage);
                }
            }
        }
    }

    private void copyFolder(OverthereFile sourceFile, TFile targetFolder) {
        for (OverthereFile overthereFile : sourceFile.listFiles()) {
            if (overthereFile.isFile()) {
                copyFile(overthereFile, new TFile(targetFolder, overthereFile.getName(), TArchiveDetector.NULL));
            } else if (overthereFile.isDirectory()) {
                TFile targetFolder1 = new TFile(targetFolder, overthereFile.getName());
                targetFolder1.mkdirs();
                copyFolder(overthereFile, targetFolder1);
            }
        }
    }

    private void copyFile(OverthereFile sourceFile, TFile targetFile) {
        InputStream inputStream = sourceFile.getInputStream();
        try {
            ByteStreams.copy(inputStream, newOutputSupplier(targetFile));
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } finally {
            Closeables.closeQuietly(inputStream);
        }
    }

    private String darName(Version version) {
        return format("%s-%s.dar", version.getApplication().getName(), version.getVersion());
    }

    private static final Logger logger = LoggerFactory.getLogger(DarPackager.class);
}
