package com.xebialabs.deployit.engine.packager.content;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.Map;

import com.xebialabs.deployit.engine.packager.manifest.ManifestBasedManifestWriter;
import com.xebialabs.deployit.engine.packager.manifest.ManifestWriter;

import de.schlichtherle.truezip.file.TArchiveDetector;
import de.schlichtherle.truezip.file.TFile;


/**
 * Represents dar package content which comes from external locations (like filesystem) and not from deployit.
 * To be used, for instance, in plugins which need to generate a dar package from files and data.
 */
public class ExternalDarContents extends DarContents {
    private final PackagingListener listener;

    private List<DarMember> deployables;

    private String version;

    private String applicationName;

    private ManifestWriter manifestWriter = new ManifestBasedManifestWriter();

    public ExternalDarContents(final PackagingListener listener, final List<DarMember> deployables, String applicationName, String version) {
        this.listener = listener;
        this.deployables = deployables;
        this.applicationName = applicationName;
        this.version = version;
    }

    @Override
    public void fillIn(final TFile file) throws IOException {

        manifestWriter.addHeader(applicationName, version);

        for (DarMember deployable : deployables) {
            listener.println("Add deployable " + deployable);

            Map<String, String> values = deployable.getValues();

            if (!deployable.getTags().isEmpty()) {
                int idx = 0;
                for (String tag : deployable.getTags()) {
                    values.put("tags-EntryValue-" + idx, tag);
                    idx++;
                }
            }

            manifestWriter.addArtifactEntry(deployable.getEntryName(), deployable.getType(), values ,deployable.getName());
        }

        listener.println(new String(manifestWriter.writeTo(new ByteArrayOutputStream()).toByteArray()));

        attachManifest(file, manifestWriter);
        copyArtifactsToDar(deployables, file);
    }

    @Override
    public String getPackageName() {
        return String.format("%s-%s.dar", applicationName, version);
    }


    private void copyArtifactsToDar(List<DarMember> deployables, TFile dar) throws IOException {
        for (DarMember deployable : deployables) {
            if (deployable.getLocation() != null) {
                TFile destArtifact = new TFile(dar, deployable.getEntryName(), TArchiveDetector.NULL);
                TFile.cp_r(deployable.getLocation(), destArtifact, TArchiveDetector.NULL);
            }
        }
    }
}
