package com.xebialabs.deployit.engine.packager.content;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.xebialabs.deployit.engine.packager.manifest.ManifestWriter;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployableArtifact;

import de.schlichtherle.truezip.file.TArchiveDetector;
import de.schlichtherle.truezip.file.TFile;

import static com.google.common.collect.Maps.newHashMap;

/**
 * Represents dar package content which comes from deployit repo
 */
public class DeployitDarContents extends DarContents {

    private DeploymentPackage deploymentPackage;

    private ManifestWriter manifestWriter;
    private Map<String,String> artifactLookup;

    public DeployitDarContents(final DeploymentPackage deploymentPackage, ManifestWriter manifestWriter) {
        this.deploymentPackage = deploymentPackage;
        this.manifestWriter = manifestWriter;
    }

    @Override
    public void fillIn(final TFile dar) throws IOException {
        attachManifest(dar, createManifest(deploymentPackage));
        copyArtifactsToDar(deploymentPackage, dar);
    }

    @Override
    public String getPackageName() {
        return String.format("%s-%s.dar", deploymentPackage.getApplication().getName(), deploymentPackage.getVersion());
    }

    private ManifestWriter createManifest(DeploymentPackage deploymentPackage) {

        artifactLookup = createArtifactLookupTabel(deploymentPackage);

        manifestWriter.addHeader(deploymentPackage.getApplication().getName(), deploymentPackage.getVersion());
        Set<? extends ConfigurationItem> deployables = deploymentPackage.getDeployables();
        for (ConfigurationItem each : deployables) {
            Map<String, String> values = getPropertyValues(deploymentPackage, each);
            String baseId;
            if (isMiddlewareResource(each)) {
                baseId = each.getName();
                manifestWriter.addResourceEntry(baseId, each.getType().toString(), values);
            } else {
                BaseDeployableArtifact artifact = (BaseDeployableArtifact) each;
                values.put("Name", each.getName());
                baseId = artifact.getFile().getName();
                manifestWriter.addArtifactEntry(baseId, each.getType().toString(), values);
            }
            writeEmbeddeds(deploymentPackage, manifestWriter, each, baseId);
        }
        return manifestWriter;
    }

    private Map<String, String> createArtifactLookupTabel(final DeploymentPackage deploymentPackage) {
        HashMap<String,String> map = newHashMap();
        Set<? extends ConfigurationItem> deployables = deploymentPackage.getDeployables();
        String baseId = deploymentPackage.getId();
        for (ConfigurationItem deployable : deployables) {
            createArtifactLookupTable(map, deployable, baseId);
        }
        return map;
    }

    private void createArtifactLookupTable(Map<String, String> map, final ConfigurationItem ci, String baseId) {

        String altId = baseId + "/";
        if(isMiddlewareResource(ci)){
            altId += ci.getName();
        } else {
            altId += ((DeployableArtifact) ci).getFile().getName();
        }
        map.put(ci.getId(), altId);

        Collection<PropertyDescriptor> propertyDescriptors = ci.getType().getDescriptor().getPropertyDescriptors();
        for (PropertyDescriptor propertyDescriptor : propertyDescriptors) {
            if(propertyDescriptor.getKind() == PropertyKind.SET_OF_CI
                    || propertyDescriptor.getKind() == PropertyKind.LIST_OF_CI){

                @SuppressWarnings("unchecked")
                Collection<ConfigurationItem> cis = (Collection<ConfigurationItem>) propertyDescriptor.get(ci);
                for (ConfigurationItem propertyCi : cis){

                    createArtifactLookupTable(map, propertyCi, altId);
                }
            } else if (propertyDescriptor.getKind() == PropertyKind.CI) {
                createArtifactLookupTable(map, (ConfigurationItem) propertyDescriptor.get(ci), altId);
            }
        }
    }


    private void writeEmbeddeds(final DeploymentPackage deploymentPackage, final ManifestWriter manifestWriter, final ConfigurationItem ci, String baseId) {

        Collection<PropertyDescriptor> propertyDescriptors = ci.getType().getDescriptor().getPropertyDescriptors();
        for (PropertyDescriptor propertyDescriptor : propertyDescriptors) {
            if(propertyDescriptor.getKind() == PropertyKind.SET_OF_CI &&
                    propertyDescriptor.getReferencedType().getDescriptor().isAssignableTo(EmbeddedDeployable.class)){

                @SuppressWarnings("unchecked")
                Collection<ConfigurationItem> embeddeds = (Collection<ConfigurationItem>) propertyDescriptor.get(ci);
                for (ConfigurationItem embedded : embeddeds){

                    Map<String, String> values = getPropertyValues(deploymentPackage, embedded);
                    String entryName = baseId + "/" + embedded.getName();
                    manifestWriter.addResourceEntry(entryName, embedded.getType().toString(), values);

                    writeEmbeddeds(deploymentPackage, manifestWriter, embedded, entryName);
                }

            }
        }
    }

    private void copyArtifactsToDar(DeploymentPackage deploymentPackage, TFile dar) throws IOException {
        // copy the artifacts to the exported DAR
        for (Deployable each : deploymentPackage.getDeployables()) {
            if (!isMiddlewareResource(each)) {
                BaseDeployableArtifact artifact = (BaseDeployableArtifact) each;
                File srcArtifact = new File(artifact.getFile().getPath());
                TFile destArtifact = new TFile(dar, artifact.getFile().getName(), TArchiveDetector.NULL);
                TFile.cp_r(srcArtifact, destArtifact, TArchiveDetector.NULL);
            }
        }
    }

    @SuppressWarnings("unchecked")
    private Map<String, String> getPropertyValues(DeploymentPackage deploymentPackage, ConfigurationItem ci) {
        Map<String, String> valueMap = newHashMap();
        for (PropertyDescriptor propertyDescriptor : ci.getType().getDescriptor().getPropertyDescriptors()) {

            if (propertyDescriptor.isHidden()) {
                continue;
            }

            if ("placeholders".equals(propertyDescriptor.getName())) {
                continue;
            }

            Object value = propertyDescriptor.get(ci);
            if (value == null)
                continue;
            switch (propertyDescriptor.getKind()) {
                case BOOLEAN:
                case INTEGER:
                case STRING:
                case ENUM:
                    valueMap.put(propertyDescriptor.getName(), String.valueOf(value));
                    break;
                case CI:
                    valueMap.put(propertyDescriptor.getName(), relativeReference(deploymentPackage, (BaseConfigurationItem) value));
                    break;
                case SET_OF_STRING:
                case LIST_OF_STRING:
                    int strCount = 1;
                    for (String s : (Collection<String>) value) {
                        valueMap.put(propertyDescriptor.getName() + "-EntryValue-" + strCount++, s);
                    }
                    break;
                case SET_OF_CI:
                case LIST_OF_CI:
                    int ciCount = 1;
                    for (ConfigurationItem ciEntry : (Collection<ConfigurationItem>) value) {
                        valueMap.put(propertyDescriptor.getName() + "-EntryValue-" + ciCount++, relativeReference(deploymentPackage, ciEntry));
                    }
                    break;
                case MAP_STRING_STRING:
                    for (Map.Entry<String, String> entry : ((Map<String, String>) value).entrySet()) {
                        valueMap.put(propertyDescriptor.getName() + "-" + entry.getKey(), entry.getValue());
                    }
                    break;
            }
        }
        return valueMap;
    }

    private String relativeReference(final DeploymentPackage deploymentPackage, final ConfigurationItem ciEntry) {
        int offset = deploymentPackage.getId().length() + 1;
        String substitute = artifactLookup.get(ciEntry.getId());
        return substitute.substring(offset);
    }

    private boolean isMiddlewareResource(ConfigurationItem ci) {
        return !ci.getType().getDescriptor().isAssignableTo(DeployableArtifact.class);
    }
}
