package com.xebialabs.deployit.engine.tasker

import akka.actor.{Props, Cancellable, Actor}
import concurrent.duration._
import java.io.{IOException, FileOutputStream, ObjectOutputStream, File}
import java.lang.String
import javassist.util.proxy.ProxyObjectOutputStream
import language.postfixOps
import com.xebialabs.deployit.engine.tasker.TaskRecoveryActor.Write
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.{StepStateEvent, TaskStateEvent}
import com.xebialabs.deployit.engine.tasker.messages.TaskStateEventHandled
import org.slf4j.LoggerFactory
import scala.util.Try
import com.google.common.io.Closeables
import grizzled.slf4j.Logging

object TaskRecoveryActor {
  def props(taskId: TaskId, recoveryDir: File) = Props(classOf[TaskRecoveryActor], taskId, recoveryDir)

  case class Write(task: Task)
}

class TaskRecoveryActor(taskId: TaskId, recoveryDir: File) extends Actor with Logging with Recovery {

  import context._

  if (!recoveryDir.exists()) recoveryDir.mkdirs()

  implicit private val implRecoveryDir = recoveryDir

  def receive: Actor.Receive = ReceiveWithMdc(taskId) {
    case e @ TaskStateEvent(`taskId`, _, _, state) if !state.isFinal =>
      debug(s"Received [$e]")
      become(writeScheduled(system.scheduler.scheduleOnce(1 second, self, Write(e.t))))
      debug(s"Scheduled Recovery writer for [$taskId] for 1 second from now")
    case e @ StepStateEvent(`taskId`, _, _, _, _, _, _) =>
      debug(s"Received [$e]")
      become(writeScheduled(system.scheduler.scheduleOnce(1 second, self, Write(e.t))))
      debug(s"Scheduled Recovery writer for [$taskId] for 1 second from now")
    case e @ TaskStateEvent(`taskId`, _, oldState, state) if state.isFinal =>
      debug(s"Cleaning up recovery file for [$taskId]")
      deleteRecoveryFile(taskId)
      system.eventStream.publish(TaskStateEventHandled(taskId, oldState, state))
  }

  private def writeScheduled(token: Cancellable): Actor.Receive = ReceiveWithMdc(taskId) {
    case e @ TaskStateEvent(`taskId`, _, oldState, state) if state.isFinal =>
      token.cancel()
      debug(s"Canceled writing recovery file for [$taskId]")
      deleteRecoveryFile(taskId)
      context.system.eventStream.publish(TaskStateEventHandled(taskId, oldState, state))
    case Write(task) =>
      debug(s"Writing recovery file for: [$task]")
      createRecoveryFile(task)
      become(receive)
    case e @ StepStateEvent =>
  }
}

trait Recovery {

  private[this] val log = LoggerFactory.getLogger(getClass)

  def createRecoveryFile(task: Task)(implicit recoveryDir: File) = {
    val id: String = task.getId
    var os: ObjectOutputStream = null
    try {
      val file = recoveryFile(task.getId)
      os = new ProxyObjectOutputStream(new FileOutputStream(file))
      os.writeObject(task)
      file
    } catch {
      case e: IOException =>
        log.error(s"Could not write recovery file for [$id]", e)
        throw e
    } finally {
      Try(Closeables.close(os, true))
    }
  }

  def deleteRecoveryFile(id: String)(implicit recoveryDir: File) = recoveryFile(id).delete()

  private def recoveryFile(id: String)(implicit recoveryDir: File): File = new File(recoveryDir, s"$id.task")
}
