package com.xebialabs.deployit.engine.tasker

import akka.actor._
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.{BlockDone, BlockStateChanged}
import com.xebialabs.deployit.engine.tasker.messages.{Abort, Start, Stop}
import com.xebialabs.deployit.engine.api.execution.BlockExecutionState

object BlockExecutingActor {
  def props(task: Task, block: CompositeBlock, ctx: TaskExecutionContext) = {
    if (block.isInstanceOf[StepBlock]) throw new IllegalArgumentException("I cannot handle Step Blocks...")
    Props(classOf[BlockExecutingActor], task, block, ctx)
  }

  case class BlockDone(taskId: TaskId, block: Block)

  case class BlockStateChanged(taskId: TaskId, block: Block, oldState: BlockExecutionState, state: BlockExecutionState)

}

class BlockExecutingActor(task: Task, b: CompositeBlock, ctx: TaskExecutionContext) extends BaseExecutionActor {

  import context._
  import BlockExecutionState._

  var block: CompositeBlock = b
  val taskId = task.getId

  def updateStateAndNotify(newState: BlockExecutionState) {
    val oldState: BlockExecutionState = block.state
    if (oldState != newState) {
      block = block.newState(newState)
      val msg: BlockStateChanged = BlockStateChanged(task.getId, block, oldState, newState)
      log.debug(s"Sending BlockStateChange($oldState->$newState) message for [${block.id}]")
      parent ! msg
    }
  }

  def receive: Actor.Receive = {
    case Start(`taskId`) if Set(PENDING, STOPPED, ABORTED, FAILED).contains(block.state) => block match {
      case parallelBlock @ ParallelBlock(id, description, blocks) =>
        log.info(s"Processing ParallelBlock[$id] of task [${task.getId}]")
        updateStateAndNotify(BlockExecutionState.EXECUTING)
        executeParallelBlock(blocks)
      case serialBlock @ SerialBlock(id, description, blocks) =>
        log.info(s"Processing SerialBlock[$id] of task [${task.getId}]")
        updateStateAndNotify(BlockExecutionState.EXECUTING)
        executeSerialBlock(blocks)
    }

  }

  def executeParallelBlock(blocks: Seq[Block]) {
    val blocksWithActors: Seq[(Block, ActorRef)] = blocks.filterNot(_.state == DONE).map(b => (b, createChildForBlock(b, ctx.contextFor(b))))
    become(waitForParallelBlocksCompleted(blocksWithActors))
    blocksWithActors.foreach {
      case (_, actor) => actor ! Start(taskId)
    }
  }

  def executeSerialBlock(blocks: Seq[Block]) {
    val head: Block = blocks.head
    head.state match {
      case DONE =>
        executeSerialBlock(blocks.tail)
      case _ =>
        val child: ActorRef = createChildForBlock(head, ctx)
        become(waitForSerialBlockCompleted(blocks, child))
        child ! Start(taskId)
    }
  }

  def createChildForBlock(b: Block, ctx: TaskExecutionContext): ActorRef = child(b.id.toBlockId) match {
    case Some(ref) => ref
    case None => b match {
      case sb: StepBlock => createChild(StepBlockExecutingActor.props(task, sb, ctx), sb.id.toBlockId)
      case cb: CompositeBlock => createChild(BlockExecutingActor.props(task, cb, ctx), cb.id.toBlockId)
    }
  }

  def waitForParallelBlocksCompleted(blocks: Seq[(Block, ActorRef)]): Actor.Receive = {
    case BlockDone(`taskId`, doneBlock: Block) =>
      log.debug(s"Received BlockDone message for [${doneBlock.id}]")
      val withoutCompletedBlock: Seq[(Block, ActorRef)] = blocks.filterNot(_._1.id == doneBlock.id)
      updateStateAndNotify(block.determineNewState(doneBlock, withoutCompletedBlock.map(_._1)))
      if (withoutCompletedBlock.isEmpty) {
        notifyAndDie()
      } else {
        become(waitForParallelBlocksCompleted(withoutCompletedBlock))
      }
    case BlockStateChanged(`taskId`, changedBlock: Block, oldState, state) =>
      log.debug(s"Received BlockStateChange($oldState->$state) message for [${block.id}]")
      updateStateAndNotify(block.determineNewState(changedBlock, blocks.map(_._1)))
      become(waitForParallelBlocksCompleted(blocks))
    case s @ Stop(`taskId`) =>
      log.debug(s"Received Stop($taskId) message, stopping sub-block")
      blocks.foreach(t => t._2 ! s)
    case m @ Abort(`taskId`) =>
      log.debug(s"Received [$m] message, aborting sub-block")
      blocks.foreach(t => t._2 ! m)
  }

  def waitForSerialBlockCompleted(blocks: Seq[Block], currentBlockActor: ActorRef): Actor.Receive = {
    case BlockDone(`taskId`, doneBlock: Block) =>
      log.debug(s"Received BlockDone message for [${doneBlock.id}]")
      val withoutCompletedBlock: Seq[Block] = blocks.tail
      updateStateAndNotify(block.determineNewState(doneBlock, withoutCompletedBlock))
      if (withoutCompletedBlock.isEmpty || doneBlock.state != DONE) {
        notifyAndDie()
      } else {
        executeSerialBlock(withoutCompletedBlock)
      }
    case BlockStateChanged(`taskId`, changedBlock: Block, oldState, state) =>
      log.debug(s"Received BlockStateChange($oldState->$state) message for [${changedBlock.id}]")
      updateStateAndNotify(block.determineNewState(changedBlock, blocks))
      become(waitForSerialBlockCompleted(blocks, currentBlockActor))
    case s @ Stop(`taskId`) =>
      log.debug(s"Received Stop($taskId) message, stopping sub-block")
      currentBlockActor ! s
    case m @ Abort(`taskId`) =>
      log.debug(s"Received [$m] message, aborting sub-block")
      currentBlockActor ! m

  }

  def notifyAndDie() {
    if (block.state == DONE) {
      log.debug("All is [EXECUTED], seppuku it is...")
      self ! PoisonPill
    } else {
      become(receive)
    }
    log.info(s"All sub blocks of [${block.id}] completed, notifying parent")
    parent ! BlockDone(taskId, block)
  }

}
