package com.xebialabs.deployit.deployment.planner;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;
import java.util.Set;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Collections2;
import com.google.common.collect.ListMultimap;

import com.xebialabs.deployit.deployment.orchestrator.OrchestratorRegistry;
import com.xebialabs.deployit.engine.spi.orchestration.*;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.Deployed;

import static com.google.common.collect.Lists.newArrayList;

public class DeploymentPlanner implements Planner {
    private Set<Method> contributors;
    private ListMultimap<Operation, Method> typeContributors;
    private List<Method> preProcessors;
    private List<Method> postProcessors;
    private Repository repository;

    private DeploymentPlanner() {
    }

    @Override
    public Plan plan(DeltaSpecification spec) {
        Orchestration orchestration = orchestrate(spec);
        Plan plan = new Plan();
        plan.getStepsWithDelta().addAll(preProcessPlan(spec).getOrderedSteps());
        plan.getStepsWithDelta().addAll(resolvePlan(orchestration, spec, plan));
        plan.getStepsWithDelta().addAll(postProcessPlan(spec).getOrderedSteps());
        return plan;
    }

    private StepPlan postProcessPlan(DeltaSpecification spec) {
        return processPlan(spec, postProcessors);
    }

    private StepPlan preProcessPlan(DeltaSpecification spec) {
        return processPlan(spec, preProcessors);
    }

    private StepPlan processPlan(DeltaSpecification spec, List<Method> processors) {
        StepPlan steps = new StepPlan();
        for (Method processor : processors) {
            try {
                Object o = processor.getDeclaringClass().newInstance();
                addResultingStepToCollector(processor.invoke(o, spec), steps, processor);
            } catch (InstantiationException e) {
                throw new PlannerException(e);
            } catch (IllegalAccessException e) {
                throw new PlannerException(e);
            } catch (InvocationTargetException e) {
                throw handleInvocationTargetException(e);
            }
        }
        return steps;
    }

    private List<Plan.StepWithDelta> resolvePlan(Orchestration orchestration, DeltaSpecification spec, Plan plan) {
        if (orchestration instanceof ParallelOrchestration) {
            return resolveParallelPlan((ParallelOrchestration) orchestration, spec, plan);
        } else if (orchestration instanceof SerialOrchestration) {
            return resolveSerialPlan((SerialOrchestration) orchestration, spec, plan);
        } else {
            StepPlan steps = new StepPlan();
            orderedResolution((InterleavedOrchestration) orchestration, steps, spec, plan);
            return steps.getOrderedSteps();
        }
    }

    private List<Plan.StepWithDelta> resolveSerialPlan(SerialOrchestration orchestration, DeltaSpecification spec, Plan plan) {
        return resolvePlans(orchestration.getPlans(), spec, plan);
    }

    /*
      * Once we support parallel, this implementation changes.
      */
    private List<Plan.StepWithDelta> resolveParallelPlan(ParallelOrchestration orchestration, DeltaSpecification spec, Plan plan) {
        return resolvePlans(orchestration.getPlans(), spec, plan);
    }

    private List<Plan.StepWithDelta> resolvePlans(List<Orchestration> orchestrations, DeltaSpecification spec, Plan plan) {
        List<Plan.StepWithDelta> resolvedPlans = newArrayList();
        for (Orchestration subplan : orchestrations) {
            resolvedPlans.addAll(resolvePlan(subplan, spec, plan));
        }
        return resolvedPlans;
    }


    private Orchestration orchestrate(DeltaSpecification spec) {
        String orchestratorName = spec.getDeployedApplication().getOrchestrator();
        Orchestrator orchestrator = OrchestratorRegistry.getOrchestrator(orchestratorName);
        Preconditions.checkNotNull(orchestrator, "Could not find a registered orchestrator with name [%s]", orchestratorName);
        return orchestrator.orchestrate(spec);
    }

    private void orderedResolution(InterleavedOrchestration orchestration, StepPlan planBuilder, DeltaSpecification spec, Plan plan) {
        DeploymentPlanningContext context = new DefaultDeploymentPlanningContext(planBuilder, spec.getDeployedApplication(), repository, plan);
        callTypeContributors(typeContributors, orchestration, planBuilder, context);
        callContributors(contributors, orchestration, context);
    }

    private void callTypeContributors(ListMultimap<Operation, Method> typeContributors, InterleavedOrchestration plan, StepPlan planBuilder, DeploymentPlanningContext context) {
        if (typeContributors == null) return;
        for (Delta dOp : plan.getDeltas()) {
            planBuilder.setDeltaUnderPlanning(dOp);
            List<Method> methods = typeContributors.get(dOp.getOperation());
            @SuppressWarnings("rawtypes")
            Deployed deployed = getActiveDeployed(dOp);
            for (Method method : methods) {
                Type type = Type.valueOf(method.getDeclaringClass());
                if (type.equals(deployed.getType())) {
                    invokeTypeContributer(context, dOp, method);
                } else {
                    Descriptor descriptor = DescriptorRegistry.getDescriptor(deployed.getType());
                    if (descriptor.isAssignableTo(type)) {
                        invokeTypeContributer(context, dOp, method);
                    }
                }
            }
        }
    }

    private void callContributors(Set<Method> methods, InterleavedOrchestration plan, DeploymentPlanningContext context) {
        if (methods == null) return;
        Deltas deltas = new Deltas(plan.getDeltas());
        for (Method method : methods) {
            try {
                Object contributorInstance = method.getDeclaringClass().newInstance();
                method.invoke(contributorInstance, deltas, context);
            } catch (InstantiationException e) {
                throw new PlannerException(e);
            } catch (IllegalAccessException e) {
                throw new PlannerException(e);
            } catch (InvocationTargetException e) {
                throw handleInvocationTargetException(e);
            }
        }
    }

    private Deployed<?, ?> getActiveDeployed(Delta dOp) {
        if (dOp.getOperation() == Operation.DESTROY) {
            return dOp.getPrevious();
        }
        return dOp.getDeployed();
    }

    private void invokeTypeContributer(DeploymentPlanningContext planContext, Delta delta, Method method) {
        try {
            if (method.getParameterTypes().length == 2) {
                method.invoke(getActiveDeployed(delta), planContext, delta);
            } else {
                method.invoke(getActiveDeployed(delta), planContext);
            }
        } catch (IllegalAccessException e) {
            throw new PlannerException(e);
        } catch (InvocationTargetException e) {
            throw handleInvocationTargetException(e);
        }
    }

    private PlannerException handleInvocationTargetException(InvocationTargetException e) {
        Throwable cause = e.getCause();
        if (cause != null) {
            return new PlannerException(cause);
        } else {
            return new PlannerException(e);
        }
    }

    @SuppressWarnings("unchecked")
    private void addResultingStepToCollector(Object result, StepPlan steps, Method method) {
        if (result == null) return;
        else if (result instanceof List) {
            steps.addAll(Collections2.transform((List) result, new Function<Object, Step>() {
                @Override
                public Step apply(Object input) {
                    return StepAdapter.wrapIfNeeded(input);
                }
            }));
        } else steps.add(StepAdapter.wrapIfNeeded(result));
    }

    public static class DeploymentPlannerBuilder {
        private DeploymentPlanner planner;

        public DeploymentPlannerBuilder() {
            this.planner = new DeploymentPlanner();
        }

        public DeploymentPlannerBuilder typeContributors(ListMultimap<Operation, Method> typeContributors) {
            planner.typeContributors = typeContributors;
            return this;
        }

        public DeploymentPlannerBuilder contributors(Set<Method> contributors) {
            planner.contributors = contributors;
            return this;
        }

        public DeploymentPlannerBuilder preProcessors(List<Method> preProcessors) {
            planner.preProcessors = preProcessors;
            return this;
        }

        public DeploymentPlannerBuilder postProcessors(List<Method> postProcessors) {
            planner.postProcessors = postProcessors;
            return this;
        }

        public DeploymentPlannerBuilder repository(Repository repository) {
            planner.repository = repository;
            return this;
        }

        public DeploymentPlanner build() {
            return planner;
        }
    }

    @SuppressWarnings("serial")
    public static class PlannerException extends RuntimeException {
        public PlannerException() {
        }

        public PlannerException(String message) {
            super(message);
        }

        public PlannerException(String message, Object... params) {
            super(String.format(message, params));
        }

        public PlannerException(String message, Throwable cause) {
            super(message, cause);
        }

        public PlannerException(Throwable cause) {
            super(cause);
        }
    }
}
