package com.xebialabs.deployit.deployment.rules;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Comparator;
import java.util.Set;
import java.util.TreeSet;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;

import com.xebialabs.deployit.plugin.api.deployment.planning.*;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.reflect.Type;

import nl.javadude.scannit.Scannit;

import static com.google.common.base.Preconditions.checkArgument;

public class JavaBasedRuleBuilder {

    void findDeployedRules() {
        registerDeployedContributors(Create.class, Operation.CREATE);
        registerDeployedContributors(Modify.class, Operation.MODIFY);
        registerDeployedContributors(Destroy.class, Operation.DESTROY);
        registerDeployedContributors(Noop.class, Operation.NOOP);
    }

    void findContributors() {
        Set<Method> contributors = new TreeSet<Method>(new HierarchyClassMethodNameComparator());
        contributors.addAll(checkContributors(Scannit.getInstance().getMethodsAnnotatedWith(Contributor.class)));
        for (Method contributor : contributors) {
//            new ContributorInvokerRule(contributor);
        }
    }

    private void registerDeployedContributors(Class<? extends Annotation> annotation, Operation operation) {
        Set<Method> typeContributors = new TreeSet<Method>(new HierarchyClassMethodNameComparator());
        typeContributors.addAll(checkDeployedContributors(Scannit.getInstance().getMethodsAnnotatedWith(annotation)));
        for (Method typeContributor : typeContributors) {
            Type t = Type.valueOf(typeContributor.getDeclaringClass());
//            RuleStore.registerRule(new DeployedInvokerRule(t, typeContributor, operation));
        }
    }

    private Set<Method> checkDeployedContributors(Set<Method> deployedContributors) {
        for (Method c : deployedContributors) {
            checkArgument(c.getReturnType().equals(void.class), "DeployedContributor %s should have void return type.", c);
            Class<?>[] parameterTypes = c.getParameterTypes();
            checkArgument(parameterTypes.length <= 2 && parameterTypes.length >= 1, "DeployedContributor %s should take 1 or 2 parameters.", c);
            checkArgument(parameterTypes[0].equals(DeploymentPlanningContext.class), "DeployedContributor %s should take %s as first parameter.", c, DeploymentPlanningContext.class);
            if (parameterTypes.length == 2) {
                checkArgument(parameterTypes[1].equals(Delta.class), "DeployedContributor %s should take %s as first parameter.", c, Delta.class);
            }
        }
        return deployedContributors;
    }

    private Set<Method> checkContributors(Set<Method> contributors) {
        for (Method contributor : contributors) {
            checkArgument(contributor.getReturnType().equals(void.class), "Contributor %s should have void return type.", contributor);
            Class<?>[] parameterTypes = contributor.getParameterTypes();
            checkArgument(parameterTypes.length == 2, "Contributor %s should take 2 parameters.", contributor);
            checkArgument(parameterTypes[0].equals(Deltas.class), "Contributor %s should take %s as first parameter.", contributor, Deltas.class);
            checkArgument(parameterTypes[1].equals(DeploymentPlanningContext.class), "Contributor %s should take %s as second parameter.", contributor, DeploymentPlanningContext.class);
        }
        return contributors;
    }


    /**
     * - First compare on the Class hierarchy, subclasses before superclasses
     * - Then on the class name
     * - Then on the method name
     */
    public static class HierarchyClassMethodNameComparator implements Comparator<Method> {

        @Override
        public int compare(Method method, Method method1) {
            Class<?> class1 = method.getDeclaringClass();
            Class<?> class2 = method1.getDeclaringClass();

            if (isSuperClass(class1, class2)) {
                return -1;
            } else if (isSuperClass(class2, class1)) {
                return 1;
            } else {
                String name1 = class1.getName();
                String name2 = class2.getName();
                int nameComparison = name1.compareTo(name2);
                if (nameComparison == 0) {
                    return method.getName().compareTo(method1.getName());
                } else {
                    return nameComparison;
                }
            }
        }

        boolean isSuperClass(Class<?> c1, Class<?> c2) {
            return c1.isAssignableFrom(c2) && !c1.getName().equals(c2.getName());
        }
    }

}
