package com.xebialabs.deployit.deployment.planner;

import java.util.Collection;
import java.util.HashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Function;
import com.google.common.collect.Iterables;

import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener;
import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentStep;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static java.lang.String.format;

public class DefaultDeploymentPlanningContext implements DeploymentPlanningContext {

    private static final Function<DeploymentStep, Step> adapt = new Function<DeploymentStep, Step>() {
        @Override
        public Step apply(DeploymentStep input) {
            return StepAdapter.wrapIfNeeded(input);
        }
    };

    private final HashMap<String, Object> attributes = newHashMap();
    private DeployedApplication deployedApplication;
    private Repository readOnlyRepository;
    private StepPlan plan;

    public DefaultDeploymentPlanningContext(DeployedApplication deployedApplication, Repository readOnlyRepository, StepPlan plan) {
        this.deployedApplication = deployedApplication;
        this.readOnlyRepository = readOnlyRepository;
        this.plan = plan;
    }

    @Override
    public void addStep(DeploymentStep step) {
        addStep(adapt.apply(step));
    }

    @Override
    public void addSteps(DeploymentStep... steps) {
        addSteps(Iterables.transform(newArrayList(steps), adapt));
    }

    @Override
    public void addSteps(Collection<DeploymentStep> steps) {
        addSteps(Iterables.transform(steps, adapt));
    }

    @Override
    public void addStep(Step step) {
        logger.trace("Adding step [{} - {} ({})]", step.getOrder(), step.getDescription(), step.getClass().getSimpleName());
        plan.addStep(step);
    }

    @Override
    public void addSteps(Step... steps) {
        logger.trace("Adding steps");
        for (Step step : steps) {
            logger.trace("[{} - {} ({})]", step.getOrder(), step.getDescription(), step.getClass().getSimpleName());
        }
        plan.addSteps(newArrayList(steps));
    }

    @Override
    public void addSteps(Iterable<Step> steps) {
        logger.trace("Adding steps");
        for (Step step : steps) {
            logger.trace("[{} - {} ({})]", step.getOrder(), step.getDescription(), step.getClass().getSimpleName());
        }
        plan.addSteps(newArrayList(steps));
    }

    @Override
    public void addCheckpoint(Step step, Delta delta) {
        plan.getCheckpoints().add(new StepPlan.Checkpoint(delta, step));
    }

    @Override
    public void addCheckpoint(final Step step, final Delta delta, final Operation overrideOperation) {
        if (delta.getOperation() != Operation.MODIFY) {
            checkArgument(
                delta.getOperation() == overrideOperation,
                format("The delta's operation (%s) and overrideOperation (%s) should match in non-modify scenarios", delta.getOperation(), overrideOperation));
        }
        plan.getCheckpoints().add(new StepPlan.Checkpoint(delta, step, overrideOperation));
    }

    @Override
    public void addCheckpoint(Step step, Iterable<Delta> deltas) {
        for (Delta delta : deltas) {
            plan.getCheckpoints().add(new StepPlan.Checkpoint(delta, step));
        }
    }

    @Override
    public void addStepWithCheckpoint(final Step step, final Delta delta) {
        addStep(step);
        addCheckpoint(step, delta);
    }

    @Override
    public void addStepWithCheckpoint(final Step step, final Delta delta, final Operation overrideOperation) {
        addStep(step);
        addCheckpoint(step, delta, overrideOperation);
    }

    @Override
    public void addStepWithCheckpoint(final Step step, final Iterable<Delta> deltas) {
        addStep(step);
        addCheckpoint(step, deltas);
    }

    @Override
    public Object getAttribute(String param) {
        return attributes.get(param);
    }

    @Override
    public void setAttribute(String param, Object value) {
        attributes.put(param, value);
        if (value instanceof ExecutionStateListener) {
            plan.getListeners().add((ExecutionStateListener) value);
        }
    }

    @Override
    public DeployedApplication getDeployedApplication() {
        return deployedApplication;
    }

    @Override
    public Repository getRepository() {
        return readOnlyRepository;
    }

    public void setDeltaUnderPlanning(Delta delta) {
        this.plan.setDeltaUnderPlanning(delta);
    }

    private static final Logger logger = LoggerFactory.getLogger(DefaultDeploymentPlanningContext.class);
}
