package com.xebialabs.deployit.deployment.rules

import java.io.{InputStream, File}
import scala.xml._
import grizzled.slf4j.Logging
import com.xebialabs.deployit.plugin.api.reflect.Type
import javax.xml.validation.SchemaFactory
import javax.xml.XMLConstants._
import java.lang.Thread._
import javax.xml.transform.stream.StreamSource
import org.xml.sax.SAXParseException
import scalax.io.{Resource, Output}

object XmlReaderWriter extends Logging {
  def loadFile(file: File, loader: Elem => Unit) {
    if (file.exists()) {
      info("Loading file [" + file + "].")
      loader(XML.loadFile(file))
    } else {
      warn("Ignoring file [" + file + "] as it does not exist")
    }
  }

  def loadStream(stream: InputStream, loader: Elem => Unit) {
    info("Loading file from stream.")
    loader(XML.load(stream))
  }

  def loadFileWithSchema(file: File, schemaName: String, loader: Elem => Unit) {
    loadFile(file, (e: Elem) => {
      try {
        validateFile(file, schemaName)
      } catch {
        case e: SAXParseException => throw new Exception("Cannot load invalid XML file", e)
      }
      info("Reading file [" + file + "].")
      loader(XML.loadFile(file))
    })
  }

  def validateFile(file: File, schemaName: String) {
    info("Validating file [" + file + "]")
    val schemaFactory = SchemaFactory.newInstance(W3C_XML_SCHEMA_NS_URI)
    val schema = schemaFactory.newSchema(currentThread().getContextClassLoader.getResource(schemaName))
    val validator = schema.newValidator()
    validator.validate(new StreamSource(file))
  }

  def writeFile(file: File, generator: => Elem) {
    val rootElem: Elem = generator
    if (file.exists()) file.delete()
    val output: Output = Resource.fromFile(file)
    output.write("<?xml version='1.0' encoding='UTF-8'?>\n" + new PrettyPrinter(200, 2).format(rootElem))
  }

  def falseDefault(n: NodeSeq): Boolean = n.headOption.exists(_.text.toBoolean)

  def trueDefault(n: NodeSeq): Boolean = booleanOption(n).getOrElse(true)

  def booleanOption(n: NodeSeq): Option[Boolean] = n.headOption.map(_.text.toBoolean)

  implicit def nodeSeqToType(n: NodeSeq): Type = Type.valueOf(n.text)

  implicit def nodeSeqToOptionType(n: NodeSeq): Option[Type] = n.headOption.map(x => Type.valueOf(x.text))

  implicit def nodeSeqToString(n: NodeSeq): String = n.text

  implicit def nodeSeqToOptionString(n: NodeSeq): Option[String] = n.headOption.map(_.text)


}
