package com.xebialabs.deployit.deployment.rules

import com.xebialabs.deployit.plugin.api.deployment.planning._
import com.xebialabs.deployit.plugin.api.deployment.specification.{Deltas, Delta, Operation}
import nl.javadude.scannit.Scannit
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.google.common.base.Preconditions._
import java.lang.reflect.Method
import scala.collection.immutable.TreeSet
import java.lang.annotation.Annotation
import collection.convert.wrapAll._
import java.util

class JavaBasedRuleBuilder(ruleStore: RuleStore) {
  private[rules] def findDeployedRules() {
    registerDeployedContributors(classOf[Create], Operation.CREATE)
    registerDeployedContributors(classOf[Modify], Operation.MODIFY)
    registerDeployedContributors(classOf[Destroy], Operation.DESTROY)
    registerDeployedContributors(classOf[Noop], Operation.NOOP)
  }

  private[rules] def findContributors() {
    implicit val ordering = new ClassHierarchyMethodOrdering
    val annotatedWith: Set[Method] = Scannit.getInstance.getMethodsAnnotatedWith(classOf[Contributor]).toSet
    val contributors: Set[Method] = TreeSet().++(checkContributors(annotatedWith))
    for (contributor <- contributors) {
      ruleStore.registerRule(new JavaContributorInvokerRule(contributor))
    }
  }

  private def registerDeployedContributors(annotation: Class[_ <: Annotation], operation: Operation) {
    implicit val ordering = new ClassHierarchyMethodOrdering
    val typeContributors: Set[Method] = TreeSet().++(checkDeployedContributors(Scannit.getInstance.getMethodsAnnotatedWith(annotation).toSet))
    for (typeContributor <- typeContributors) {
      val t: Type = Type.valueOf(typeContributor.getDeclaringClass)
      ruleStore.registerRule(new JavaDeployedInvokerRule(t, typeContributor, operation))
    }
  }

  private def checkDeployedContributors(deployedContributors: Set[Method]): Set[Method] = {
    for (c <- deployedContributors) {
      checkArgument(c.getReturnType == classOf[Unit], "DeployedContributor %s should have void return type.", c)
      val parameterTypes: Array[Class[_]] = c.getParameterTypes
      checkArgument(parameterTypes.length <= 2 && parameterTypes.length >= 1, "DeployedContributor %s should take 1 or 2 parameters.", c)
      checkArgument(parameterTypes(0) == classOf[DeploymentPlanningContext], "DeployedContributor %s should take %s as first parameter.", c, classOf[DeploymentPlanningContext])
      if (parameterTypes.length == 2) {
        checkArgument(parameterTypes(1) == classOf[Delta], "DeployedContributor %s should take %s as first parameter.", c, classOf[Delta])
      }
    }
    deployedContributors
  }

  private def checkContributors(contributors: Set[Method]): Set[Method] = {
    for (contributor <- contributors) {
      checkArgument(contributor.getReturnType == classOf[Unit], "Contributor %s should have void return type.", contributor)
      val parameterTypes: Array[Class[_]] = contributor.getParameterTypes
      checkArgument(parameterTypes.length == 2, "Contributor %s should take 2 parameters.", contributor)
      checkArgument(parameterTypes(0) == classOf[Deltas], "Contributor %s should take %s as first parameter.", contributor, classOf[Deltas])
      checkArgument(parameterTypes(1) == classOf[DeploymentPlanningContext], "Contributor %s should take %s as second parameter.", contributor, classOf[DeploymentPlanningContext])
    }
    contributors
  }

  /**
   * - First compare on the Class hierarchy, subclasses before superclasses
   * - Then on the class name
   * - Then on the method name
   */
  class ClassHierarchyMethodOrdering extends Ordering[Method] {
    def compare(method: Method, method1: Method): Int = {
      val class1: Class[_] = method.getDeclaringClass
      val class2: Class[_] = method1.getDeclaringClass
      if (isSuperClass(class1, class2)) {
        -1
      } else if (isSuperClass(class2, class1)) {
        1
      } else {
        val name1: String = class1.getName
        val name2: String = class2.getName
        val nameComparison: Int = name1.compareTo(name2)
        if (nameComparison == 0) {
          method.getName.compareTo(method1.getName)
        } else {
          nameComparison
        }
      }
    }

    private[rules] def isSuperClass(c1: Class[_], c2: Class[_]): Boolean = {
      c1.isAssignableFrom(c2) && !(c1.getName == c2.getName)
    }
  }
}
