package com.xebialabs.deployit.deployment
package rules

import com.xebialabs.deployit.plugin.api.deployment.specification.{Delta, Operation}
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.rules.{StepPostConstructContext, Scope}
import com.xebialabs.platform.script.jython.JythonContext


private[rules] class XmlPlanRule(name: String,
                                 scope: Scope,
                                 override val condition: Option[String],
                                 steps: Iterable[XmlStepDataWithCheckPoint])
  extends XmlRule(name, scope, condition, steps) with PlanRule

private[rules] class XmlDeployedRule(name: String,
                                     override val ciTypes: Iterable[Type],
                                     override val operations: Iterable[Operation],
                                     override val condition: Option[String],
                                     steps: Iterable[XmlStepDataWithCheckPoint])
  extends XmlRule(name, Scope.DEPLOYED, condition, steps) with DeployedRule

private[rules] abstract class XmlRule(name: String, scope: Scope, condition: Option[String], steps: Iterable[XmlStepDataWithCheckPoint])
  extends Rule(name, scope) with ContextFactory {

  require(steps.forall(stepData => isUnique(stepData.stepParameters)), "Step parameters must be unique")

  def isUnique(steps: Iterable[XmlParameter]) = steps.map(_.name).toSet.size == steps.size

  override def doFire(scopedObject: AnyRef, planningContext: RulePlanningContext): Unit = {
    implicit val pContext = planningContext
    implicit val jContext: JythonContext = jythonContext(scopedObject, getScope)
    implicit val stepContext = stepPostConstructContext(scope, scopedObject)
    steps.foreach { stepData =>
      val step = createStep(stepData, planningContext.getStepRegistry)
      stepData.checkpoint match {
        case Some(XmlCheckpoint(Some(operation))) if scopedObject.isInstanceOf[Delta] =>
          planningContext.addStepWithCheckpoint(step, scopedObject.asInstanceOf[Delta], operation)
        case Some(XmlCheckpoint(None)) if scopedObject.isInstanceOf[Delta] =>
          planningContext.addStepWithCheckpoint(step, scopedObject.asInstanceOf[Delta])
        case None => planningContext.addStep(step)
        case _ => throw PlannerException(message = "Checkpoint was set for a scope that is not DEPLOYED")
      }
    }
  }

  def createStep(data: XmlStepDataWithCheckPoint, stepRegistry: StepRegistry)(implicit scriptContext: JythonContext, stepContext: StepPostConstructContext): Step = {
    val xmlParamResolver = new XmlParameterResolver(stepRegistry)
    new StepFactory(stepRegistry, stepContext).step(StepData(data.stepName, xmlParamResolver.resolveXmlParameters(data.stepName, data.stepParameters)))
  }
}

sealed trait XmlParameter {
  val name: String
}

case class XmlExpressionParameter(name: String, expression: String) extends XmlParameter

case class XmlBasicParameter(name: String, value: String) extends XmlParameter

case class XmlCollectionParameter(name: String, parameters: Iterable[XmlParameter] = Nil) extends XmlParameter

sealed trait XmlStepElem

case class XmlCheckpoint(overrideOperation: Option[Operation] = None) extends XmlStepElem

case class XmlStepData(stepName: String, stepParameters: Iterable[XmlParameter] = Nil) extends XmlStepElem

object XmlStepDataWithCheckPoint {
  def apply(stepName: String, stepParameters: Iterable[XmlParameter] = Nil, checkpoint: Option[XmlCheckpoint] = None): XmlStepDataWithCheckPoint =
    XmlStepDataWithCheckPoint(XmlStepData(stepName, stepParameters), checkpoint)
}

case class XmlStepDataWithCheckPoint(stepData: XmlStepData, checkpoint: Option[XmlCheckpoint]){
  def stepName = stepData.stepName
  def stepParameters = stepData.stepParameters
}
