package com.xebialabs.deployit.deployment
package rules

import com.xebialabs.deployit.plugin.api.deployment.specification.{Delta, Operation}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.rules.{Scope, StepPostConstructContext}
import com.xebialabs.platform._
import com.xebialabs.platform.script.jython.{JythonSupport, ScriptSource}
import grizzled.slf4j.Logging


class ScriptPlanRule(name: String,
                     scope: Scope,
                     scriptSource: ScriptSource,
                     override val condition: Option[String] = None)
  extends ScriptRule(name, scope, scriptSource, condition) with PlanRule

class ScriptDeployedRule(name: String,
                         scriptSource: ScriptSource,
                         override val ciTypes: Iterable[Type],
                         override val operations: Iterable[Operation],
                         override val condition: Option[String] = None)
  extends ScriptRule(name, Scope.DEPLOYED, scriptSource, condition) with DeployedRule

abstract class ScriptRule(name: String, scope: Scope, scriptSource: ScriptSource, condition: Option[String]) extends Rule(name, scope) with ContextFactory with Logging with JythonSupport {

  override def doFire(scopedObject: AnyRef, context: RulePlanningContext) = {
    implicit val pContext = context
    executeScript(scriptSource)(jythonContext(scopedObject, getScope))
  }

  def getScript: String = scriptSource.scriptContent
}

class ScriptStepFactory(stepRegistry: StepRegistry, stepPostConstructContext: StepPostConstructContext)
  extends StepFactory(stepRegistry, stepPostConstructContext) {

  import scala.collection.convert.wrapAll._

  def stepFromScript(name: String, args: java.util.List[Any], namedParameters: java.util.Map[String, Any]) = {
    validateArguments(args.toList)
    step(StepData(name.underscoresToDashes, namedParameters.toMap.map { entry => (entry._1.underscoresToDashes, entry._2)}.toMap))
  }

  private def validateArguments(args: List[Any]) {
    if (args != null && args.nonEmpty) {
      throw new IllegalArgumentException("Only named parameters are supported")
    }
  }
}

trait ContextFactory extends Logging {

  import com.xebialabs.platform.script.jython._

  def stepPostConstructContext(scope: Scope, scopedObject: AnyRef)(implicit context: RulePlanningContext) =
    new StepPostConstructContext(scope, scopedObject, context.getDeployedApplication, context.getRepository)

  def jythonContext(scopedObject: AnyRef, scope: Scope)(implicit context: RulePlanningContext) = {
    val bindings = allBindings(scopedObject, scope)
    JythonContext(Syntactic.wrapperCodeWithLib(bindings.keySet) :+ ruleLib :+ Syntactic.loggerLib, bindings)
  }

  private def allBindings(scopedObject: AnyRef, scope: Scope)(implicit context: RulePlanningContext) = {
    scopedBindings(scopedObject, scope) ++ unscopedBindings(createStepFactory(scopedObject, scope)) ++ Bindings.xlDeployApiServices
  }

  private def scopedBindings(scopedObject: AnyRef, scope: Scope): Map[String, AnyRef] = scope match {
    case Scope.DEPLOYED => Map("delta" -> scopedObject,
      "deployed" -> scopedObject.asInstanceOf[Delta].getDeployed,
      "previousDeployed" -> scopedObject.asInstanceOf[Delta].getPrevious)
    case Scope.PLAN => Map("deltas" -> scopedObject)
    case Scope.PRE_PLAN | Scope.POST_PLAN => Map("specification" -> scopedObject)
    case _ => Map()
  }

  private def unscopedBindings(stepFactory: ScriptStepFactory)(implicit context: RulePlanningContext) = Map(
    "deployedApplication" -> context.getDeployedApplication,
    "context" -> context,
    "_internal_step_factory" -> stepFactory
  )

  private def createStepFactory(scopedObject: AnyRef, scope: Scope)(implicit context: RulePlanningContext): ScriptStepFactory = {
    new ScriptStepFactory(context.getStepRegistry, stepPostConstructContext(scope, scopedObject))
  }
}
