package com.xebialabs.deployit.deployment.rules

import java.util
import java.util.{List => JList}

import akka.actor.ActorRef
import akka.pattern.ask
import akka.util.Timeout
import com.xebialabs.deployit.deployment.orchestrator.{OrchestratorComposer, OrchestratorRegistry}
import com.xebialabs.deployit.deployment.planner._
import com.xebialabs.deployit.deployment.rules.RegistriesActor.Messages.{RuleRegistryGet, RuleRegistryReturn, StepRegistryGet, StepRegistryReturn}
import com.xebialabs.deployit.engine.spi.exception.{DeployitException, HttpResponseCodeResult}
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener
import com.xebialabs.deployit.engine.spi.orchestration._
import com.xebialabs.deployit.plugin.api.deployment.specification.{Delta, DeltaSpecification, Deltas}
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.rules.Scope
import com.xebialabs.deployit.plugin.api.services.Repository
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication
import grizzled.slf4j.Logging

import scala.collection.convert.wrapAll._
import scala.collection.mutable
import scala.concurrent.Await
import scala.concurrent.duration._

class RuleBasedPlanner(registriesActor: ActorRef) extends Planner with Logging {

  private val composer: OrchestratorComposer = new OrchestratorComposer
  private implicit val akkaAskTimeout: Timeout = 1.minute

  override def plan(spec: DeltaSpecification, repository: Repository): SerialPlan = {
    import scala.concurrent.ExecutionContext.Implicits.global
    try {
      val ruleResult = (registriesActor ? RuleRegistryGet()).mapTo[RuleRegistryReturn].map(_.ruleRegistry)
      val stepResult = (registriesActor ? StepRegistryGet()).mapTo[StepRegistryReturn].map(_.stepRegistry)
      implicit val stepRegistry = Await.result(stepResult, akkaAskTimeout.duration)
      implicit val ruleRegistry = Await.result(ruleResult, akkaAskTimeout.duration)
      createPlan(spec, repository)
    } catch {
      case pe: PlannerException =>
        throw pe
      case ex: Exception =>
        throw new PlannerException(s"An error occurred while planning: ${ex.getMessage}", ex)
    }
  }

  private[rules] def createPlan(spec: DeltaSpecification, repository: Repository)(implicit ruleRegistry: RuleRegistry, stepRegistry: StepRegistry): SerialPlan = {
    val listeners = new util.ArrayList[ExecutionStateListener]()
    val prePlan = preProcessPlan(spec, repository, listeners)
    val orchestration = doOrchestrate(spec)
    val plan = resolvePlan(orchestration, spec.getDeployedApplication, repository, listeners)
    val postPlan: Plan = postProcessPlan(spec, repository, listeners)
    new SerialPlan(orchestration.getDescription, List(prePlan, plan, postPlan), listeners)
  }

  private def doOrchestrate(spec: DeltaSpecification): Orchestration = {
    val orchestratorIds = spec.getDeployedApplication.getOrchestrator
    val orchestrators = OrchestratorRegistry.getOrchestrators(orchestratorIds)
    composer.orchestrate(orchestrators, spec)
  }

  private def preProcessPlan(spec: DeltaSpecification, repository: Repository, listeners: JList[ExecutionStateListener])(implicit ruleRegistry: RuleRegistry, stepRegistry: StepRegistry) =
    processPlan("Prepare deployment", spec, ruleRegistry.getEnabledRules(Scope.PRE_PLAN), repository, listeners)

  private def postProcessPlan(spec: DeltaSpecification, repository: Repository, listeners: JList[ExecutionStateListener])(implicit ruleRegistry: RuleRegistry, stepRegistry: StepRegistry) =
    processPlan("Finalize deployment", spec, ruleRegistry.getEnabledRules(Scope.POST_PLAN), repository, listeners)

  private def processPlan(description: String, spec: DeltaSpecification, processors: Iterable[Rule], repository: Repository, listeners: JList[ExecutionStateListener])(implicit ruleRegistry: RuleRegistry, stepRegistry: StepRegistry) = {
    val plan = new StepPlan(description, listeners)
    val ctx = new RulePlanningContext(spec.getDeployedApplication, stepRegistry, repository, plan)
    processors.foreach { rule => fireAndLog(rule, plan, spec, ctx)}
    plan
  }

  private def resolvePlan(orchestration: Orchestration, application: DeployedApplication, repository: Repository, listeners: JList[ExecutionStateListener])(implicit ruleRegistry: RuleRegistry, stepRegistry: StepRegistry): Plan = {
    orchestration match {
      case pp: ParallelOrchestration => new ParallelPlan(orchestration.getDescription, pp.getPlans.map(resolvePlan(_, application, repository, listeners)), listeners)
      case sp: SerialOrchestration => new SerialPlan(orchestration.getDescription, sp.getPlans.map(resolvePlan(_, application, repository, listeners)), listeners)
      case ip: InterleavedOrchestration => resolveInterleavedPlan(ip, application, repository, listeners)
    }
  }

  private def resolveInterleavedPlan(orchestration: InterleavedOrchestration, deployedApplication: DeployedApplication, repository: Repository, listeners: JList[ExecutionStateListener])(implicit ruleRegistry: RuleRegistry, stepRegistry: StepRegistry): Plan = {
    val stepPlan = new StepPlan(orchestration.getDescription, listeners)
    val ctx = new RulePlanningContext(deployedApplication, stepRegistry, repository, stepPlan)

    // Fire Deployed contributor rules
    val deployedRules = ruleRegistry.getEnabledRules(Scope.DEPLOYED)
    orchestration.getDeltas.foreach { delta =>
      try {
        stepPlan.setDeltaUnderPlanning(delta)
        deployedRules.foreach {
          rule => fireAndLog(rule, stepPlan, delta, ctx)
        }
      } finally {
        stepPlan.setDeltaUnderPlanning(null)
      }
    }

    // Fire plan contributor rules
    ruleRegistry.getEnabledRules(Scope.PLAN).foreach { rule => fireAndLog(rule, stepPlan, new Deltas(orchestration.getDeltas), ctx)}
    addImplicitCheckpoints(ctx, stepPlan)
    stepPlan
  }

  private def addImplicitCheckpoints(ctx: RulePlanningContext, stepPlan: StepPlan): Unit = {
    import scala.collection.convert.wrapAsScala._

    val checkpointedDeltas: mutable.Buffer[Delta] = stepPlan.getCheckpoints.map(ch => ch.getDelta)
    val deltasWithSteps = stepPlan.getStepsWithPlanningInfo.foldLeft(new DeltasWithSteps) {
      case (deltas, stepWithInfo) => stepWithInfo.getDeltas.foreach(deltas.addBinding(_, stepWithInfo.getStep)); deltas
    }
    deltasWithSteps.filterKeys(!checkpointedDeltas.contains(_)).foreach {
      case (delta, steps) =>
        val lastStep = steps.zipWithIndex.maxBy { case (step, index) => step.getOrder -> index}._1
        ctx.addCheckpoint(lastStep, delta)
    }
  }

  private def fireAndLog(rule: Rule, stepPlan: StepPlan, scopedObject: AnyRef, ctx: RulePlanningContext) = {
    try {
      stepPlan.setRuleUnderPlanning(rule.getName)
      trace(s"Checking conditions for rule $rule")
      if (rule.canFire(scopedObject, ctx)) {
        debug(s"Firing rule ${rule.getName} while deploying ${ctx.getDeployedApplication.getVersion} to ${ctx.getDeployedApplication.getEnvironment}")
        rule.doFire(scopedObject, ctx)
      }
    } catch {
      case ex: Exception =>
        error(s"Error while executing rule $rule for scopedObject: $scopedObject", ex)
        throw new PlannerException(s"Error while evaluating rule [${rule.getName}]: ${ex.getMessage}", ex)
    } finally {
      stepPlan.setRuleUnderPlanning(null)
    }
  }
}

private class DeltasWithSteps extends mutable.HashMap[Delta, mutable.Set[Step]] with mutable.MultiMap[Delta, Step] {
  override protected def makeSet: mutable.Set[Step] = new mutable.LinkedHashSet[Step]
}

@HttpResponseCodeResult(statusCode = 400)
case class PlannerException(message: String = null, cause: Throwable = null) extends DeployitException(message, cause)
