package com.xebialabs.deployit.deployment.planner;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Function;
import com.google.common.collect.ListMultimap;

import com.xebialabs.deployit.deployment.orchestrator.OrchestratorComposer;
import com.xebialabs.deployit.deployment.orchestrator.OrchestratorRegistry;
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener;
import com.xebialabs.deployit.engine.spi.orchestration.*;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;

import static com.google.common.collect.Lists.transform;

public class DeploymentPlanner implements Planner {
    private Set<Method> contributors;
    private ListMultimap<Operation, Method> typeContributors;
    private List<Method> preProcessors;
    private List<Method> postProcessors;

    private OrchestratorComposer composer = new OrchestratorComposer();

    private DeploymentPlanner() {
    }

    @Override
    public Plan plan(DeltaSpecification spec, Repository repository) {
        List<ExecutionStateListener> listeners = new ArrayList<ExecutionStateListener>();

        Plan prePlan = preProcessPlan(spec, listeners);
        Orchestration orchestrate = orchestrate(spec);
        Plan plan = resolvePlan(orchestrate, spec.getDeployedApplication(), repository, listeners);
        Plan postPlan = postProcessPlan(spec, listeners);

        return new SerialPlan(orchestrate.getDescription(), Arrays.asList(prePlan, plan, postPlan), listeners);
    }

    private StepPlan postProcessPlan(DeltaSpecification spec, List<ExecutionStateListener> listeners) {
        return processPlan("Finalize deployment", spec, postProcessors, listeners);
    }

    private StepPlan preProcessPlan(DeltaSpecification spec, List<ExecutionStateListener> listeners) {
        return processPlan("Prepare deployment", spec, preProcessors, listeners);
    }

    private StepPlan processPlan(String description, DeltaSpecification spec, List<Method> processors, List<ExecutionStateListener> listeners) {
        StepPlan steps = new StepPlan(description, listeners);
        for (Method processor : processors) {
            try {
                Object o = processor.getDeclaringClass().newInstance();
                addResultingStepToCollector(processor.invoke(o, spec), steps, processor);
            } catch (InstantiationException e) {
                throw new PlannerException(e);
            } catch (IllegalAccessException e) {
                throw new PlannerException(e);
            } catch (InvocationTargetException e) {
                throw handleInvocationTargetException(e);
            }
        }
        return steps;
    }

    private Plan resolvePlan(Orchestration orchestration, DeployedApplication deployedApplication, Repository repository, List<ExecutionStateListener> listeners) {
        if (orchestration instanceof ParallelOrchestration) {
            return resolveParallelOrchestration((ParallelOrchestration) orchestration, deployedApplication, repository, listeners);
        } else if (orchestration instanceof SerialOrchestration) {
            return resolveSerialOrchestration((SerialOrchestration) orchestration, deployedApplication, repository, listeners);
        } else {
            return resolveInterleavedOrchestration((InterleavedOrchestration) orchestration, deployedApplication, repository, listeners);
        }
    }

    private Plan resolveInterleavedOrchestration(final InterleavedOrchestration orchestration, final DeployedApplication deployedApplication, final Repository repository, final List<ExecutionStateListener> listeners) {
        StepPlan stepPlan = new StepPlan(orchestration.getDescription(), listeners);
        DefaultDeploymentPlanningContext ctx = new DefaultDeploymentPlanningContext(deployedApplication, repository, stepPlan);
        callTypeContributors(typeContributors, orchestration, ctx, stepPlan);
        callContributors(contributors, orchestration, ctx);
        return stepPlan;
    }

    private SerialPlan resolveSerialOrchestration(SerialOrchestration orchestration, DeployedApplication deployedApplication, Repository repository, List<ExecutionStateListener> listeners) {
        return new SerialPlan(orchestration.getDescription(), resolvePlans(orchestration.getPlans(), deployedApplication, repository, listeners), listeners);
    }

    private Plan resolveParallelOrchestration(ParallelOrchestration orchestration, DeployedApplication deployedApplication, Repository repository, List<ExecutionStateListener> listeners) {
        return new ParallelPlan(orchestration.getDescription(), resolvePlans(orchestration.getPlans(), deployedApplication, repository, listeners), listeners);
    }

    private List<Plan> resolvePlans(List<Orchestration> orchestrations, final DeployedApplication deployedApplication, final Repository repository, final List<ExecutionStateListener> listeners) {
        return transform(orchestrations, new Function<Orchestration, Plan>() {
            @Override
            public Plan apply(Orchestration o) {
                return resolvePlan(o, deployedApplication, repository, listeners);
            }
        });
    }

    private Orchestration orchestrate(DeltaSpecification spec) {
        List<String> orchestratorIds = spec.getDeployedApplication().getOrchestrator();
        List<Orchestrator> orchestrators = OrchestratorRegistry.getOrchestrators(orchestratorIds);
        return composer.orchestrate(orchestrators, spec);
    }

    private void callTypeContributors(ListMultimap<Operation, Method> typeContributors, InterleavedOrchestration plan, DefaultDeploymentPlanningContext context, final StepPlan stepPlan) {
        if (typeContributors == null) return;
        for (Delta dOp : plan.getDeltas()) {
            try {
                stepPlan.setDeltaUnderPlanning(dOp);
                List<Method> methods = typeContributors.get(dOp.getOperation());
                @SuppressWarnings("rawtypes")
                Deployed deployed = getActiveDeployed(dOp);
                for (Method method : methods) {
                    Type methodType = Type.valueOf(method.getDeclaringClass());
                    if (deployed.getType().instanceOf(methodType)) {
                        invokeTypeContributor(context, dOp, method);
                    }
                }
            } finally {
                stepPlan.setDeltaUnderPlanning(null);
            }
        }
    }

    private void callContributors(Set<Method> methods, InterleavedOrchestration plan, DeploymentPlanningContext context) {
        if (methods == null) return;
        Deltas deltas = new Deltas(plan.getDeltas());
        for (Method method : methods) {
            try {
                logger.trace("Invoking Contributor [{}] for [{}]", method, plan.getDeltas());
                Object contributorInstance = method.getDeclaringClass().newInstance();
                method.invoke(contributorInstance, deltas, context);
            } catch (InstantiationException e) {
                throw new PlannerException(e);
            } catch (IllegalAccessException e) {
                throw new PlannerException(e);
            } catch (InvocationTargetException e) {
                throw handleInvocationTargetException(e);
            }
        }
    }

    private Deployed<?, ?> getActiveDeployed(Delta dOp) {
        if (dOp.getOperation() == Operation.DESTROY) {
            return dOp.getPrevious();
        }
        return dOp.getDeployed();
    }

    private void invokeTypeContributor(DeploymentPlanningContext planContext, Delta delta, Method method) {
        logger.trace("Invoking Type Contributor [{}.{}] for [{}]", Type.valueOf(method.getDeclaringClass()), method.getName(), delta);
        try {
            if (method.getParameterTypes().length == 2) {
                method.invoke(getActiveDeployed(delta), planContext, delta);
            } else {
                method.invoke(getActiveDeployed(delta), planContext);
            }
        } catch (IllegalAccessException e) {
            throw new PlannerException(e);
        } catch (InvocationTargetException e) {
            throw handleInvocationTargetException(e);
        }
    }

    private PlannerException handleInvocationTargetException(InvocationTargetException e) {
        Throwable cause = e.getCause();
        if (cause != null) {
            return new PlannerException(cause);
        } else {
            return new PlannerException(e);
        }
    }

    @SuppressWarnings("unchecked")
    private void addResultingStepToCollector(Object result, StepPlan steps, Method method) {
        if (result == null) {
            return;
        } else if (result instanceof List) {
            steps.addSteps(transform((List) result, new Function<Object, Step>() {
                @Override
                public Step apply(Object input) {
                    return StepAdapter.wrapIfNeeded(input);
                }
            }));
        } else {
            steps.addStep(StepAdapter.wrapIfNeeded(result));
        }
    }

    public static class DeploymentPlannerBuilder {
        private DeploymentPlanner planner;

        public DeploymentPlannerBuilder() {
            this.planner = new DeploymentPlanner();
        }

        public DeploymentPlannerBuilder typeContributors(ListMultimap<Operation, Method> typeContributors) {
            planner.typeContributors = typeContributors;
            return this;
        }

        public DeploymentPlannerBuilder contributors(Set<Method> contributors) {
            planner.contributors = contributors;
            return this;
        }

        public DeploymentPlannerBuilder preProcessors(List<Method> preProcessors) {
            planner.preProcessors = preProcessors;
            return this;
        }

        public DeploymentPlannerBuilder postProcessors(List<Method> postProcessors) {
            planner.postProcessors = postProcessors;
            return this;
        }

        public DeploymentPlanner build() {
            return planner;
        }
    }

    @SuppressWarnings("serial")
    public static class PlannerException extends RuntimeException {
        public PlannerException() {
        }

        public PlannerException(String message) {
            super(message);
        }

        public PlannerException(String message, Object... params) {
            super(String.format(message, params));
        }

        public PlannerException(String message, Throwable cause) {
            super(message, cause);
        }

        public PlannerException(Throwable cause) {
            super(cause);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(DeploymentPlanner.class);
}
