package com.xebialabs.deployit.deployment.rules

import java.io.InputStream
import java.io.OutputStream

trait ManagedResource[T] {
  def onEnter(): T
  def onExit(t:Throwable = null): Unit
  def attempt(block: => Unit): Unit = {
    try { block } finally {}
  }
}

class ManagedOutputStream(out:OutputStream) extends ManagedResource[OutputStream] {
  def onEnter(): OutputStream = out
  def onExit(t:Throwable = null): Unit = {
    attempt(out.close())
    if (t != null) throw t
  }
}

class ManagedInputStream(in:InputStream) extends ManagedResource[InputStream] {
  def onEnter(): InputStream = in
  def onExit(t:Throwable = null): Unit = {
    attempt(in.close())
    if (t != null) throw t
  }
}

object ResourceManagement {
  def using[T <: Any](managed: ManagedResource[T])(body: T => Unit) {
    val resource = managed.onEnter()
    var exception = false
    try {
      body(resource)
    } catch {
      case e: Throwable =>
        exception = true
        managed.onExit(e)
    } finally {
      if (!exception) managed.onExit()
    }
  }

  def using[T <: Any, U <: Any](managed1: ManagedResource[T], managed2: ManagedResource[U])(body: T => U => Unit) {
    using[T](managed1) { r: T =>
      using[U](managed2) { s: U => body(r)(s) }
    }
  }

  implicit def os2managed(out:OutputStream): ManagedResource[OutputStream] = new ManagedOutputStream(out)
  implicit def is2managed(in:InputStream): ManagedResource[InputStream] = new ManagedInputStream(in)
}
