package com.xebialabs.deployit.deployment.stager

import com.xebialabs.deployit.deployment.planner.PlanSugar._
import com.xebialabs.deployit.deployment.planner._
import com.xebialabs.deployit.deployment.stager.DeploymentStager._StagingContext
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification
import com.xebialabs.deployit.plugin.api.flow._
import com.xebialabs.deployit.plugin.api.services.Repository
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact
import com.xebialabs.overthere.{OverthereConnection, OverthereFile}
import grizzled.slf4j.Logging

import scala.collection.convert.wrapAll._
import scala.collection.mutable
import scala.collection.mutable.ListBuffer
import scala.util.Try

class DeploymentStager(wrappedPlanner: Planner) extends Planner with Logging with PlanSugar {

  // Duck typing, Ric forced me to rename this from Duck to HasStagingFile ;-)
  type HasStagingTarget = {def getStagingTarget: StagingTarget}

  def plan(spec: DeltaSpecification, repository: Repository): PhasedPlan = {
    val stagingContext = new _StagingContext
    val plan = wrappedPlanner.plan(spec, repository)

    logger.info(s"Staging artifacts for plan: [${plan.getDescription}]")

    val staged = plan.phases.map(stage(_, stagingContext))

    val cleanupPlans = staged.map(_._2).flatten.toList

    val newPhases = staged.map(_._1)

    cleanupPlans match {
      case Nil =>
        plan.copy(newPhases)

      case cleanupPlan :: Nil =>
        newPhases += new PlanPhase(cleanupPlan, cleanupPlan.getDescription, cleanupPlan.getListeners, alwaysExecuted = true)
        plan.copy(newPhases)

      case plans =>
        val cleanupPlan = new ParallelPlan(cleanupPlans.head.getDescription, plans, plans.head.getListeners)
        newPhases += new PlanPhase(cleanupPlan, cleanupPlan.getDescription, cleanupPlan.getListeners, alwaysExecuted = true)
        plan.copy(newPhases)

    }
  }

  private def stage(planPhase: PlanPhase, stagingContext: _StagingContext): (PlanPhase, Option[ExecutablePlan]) = {
    val staged = stage(planPhase.plan, stagingContext)
    (planPhase.copy(plan = staged._1), staged._2)
  }

  private def stage(plan: ExecutablePlan, stagingContext: _StagingContext): (ExecutablePlan, Option[ExecutablePlan]) = {
    doStage(plan, stagingContext)
    if (stagingContext.stagingSteps.isEmpty) {
      (plan, None)
    } else {
      def stepsToPlan(steps: Iterable[Step with HasStagingTarget], descriptionPrefix: String): List[StepPlan] = {
        steps.groupBy(_.getStagingTarget).map({ case (t, s) => new StepPlan(s"$descriptionPrefix ${t.getName}", s, plan.getListeners)}).toList
      }

      val stagingPlan = new ParallelPlan("Stage artifacts", stepsToPlan(stagingContext.stagingSteps, "Staging to"), plan.getListeners)
      val cleanupPlan = new ParallelPlan("Clean up staged artifacts", stepsToPlan(stagingContext.cleanupHosts.map(t => new StagedFileCleaningStep(t)), "Clean up staged files on"), plan.getListeners)
      val planWithStaging = new SerialPlan(plan.getDescription, List(stagingPlan, plan), plan.getListeners)
      (planWithStaging, Some(cleanupPlan))
    }
  }

  private def doStage(plan: Plan, stagingContext: _StagingContext) {
    logger.debug(s"Staging for [${plan.getClass.getSimpleName}(${plan.getDescription})]")
    plan match {
      case cp: CompositePlan => cp.getSubPlans.foreach(doStage(_, stagingContext))
      case sp: StepPlan => doStage(sp, stagingContext)
    }
  }

  private def doStage(stepPlan: StepPlan, stagingContext: _StagingContext) {
    stepPlan.getSteps.withFilter(_.isInstanceOf[StageableStep]).foreach(step => doStage(step.asInstanceOf[StageableStep], stagingContext))
  }

  private def doStage(step: StageableStep, stagingContext: _StagingContext) {
    logger.debug(s"Preparing stage of artifacts for step [${step.getDescription}]")
    step.requestStaging(stagingContext)
  }
}

object DeploymentStager extends Logging {

  private[stager] class JustInTimeFile(source: Artifact) extends StagedFile {
    def get(c: OverthereConnection, ctx: ExecutionContext): OverthereFile = {
      logger.debug(s"Creating temporary file for artifact [${source.getName}]")
      val uploadedFileArtifact: OverthereFile = c.getTempFile(source.getFile.getName)

      ctx.logOutput(s"Uploading artifact [${uploadedFileArtifact.getPath}]")
      source.getFile.copyTo(uploadedFileArtifact)

      logger.debug(s"Uploaded artifact [${uploadedFileArtifact.getPath}]")
      uploadedFileArtifact
    }
  }

  private[stager] class StagingFile(source: Artifact, target: StagingTarget) extends StagedFile {

    def getSource = source

    def getTarget = target

    var stagedFilePath: Option[String] = None

    def get(c: OverthereConnection, ctx: ExecutionContext): OverthereFile = {
      stagedFilePath.map(c.getFile).getOrElse({
        val file = stageFileWithConnection(ctx, c)
        stagedFilePath = Some(file.getPath)
        file
      })
    }

    def stageFile(ctx: ExecutionContext): OverthereFile = {
      val connection: OverthereConnection = target.getConnection
      try {
        stageFileWithConnection(ctx, connection)
      } catch {
        case ex: Exception =>
          ctx.logError(s"Failed to upload [${source.getFile}] to [$connection]")
          throw ex
      } finally {
        connection.close()
      }
    }

    private[this] def stageFileWithConnection(ctx: ExecutionContext, connection: OverthereConnection): OverthereFile = {
      ctx.logOutput(s"Staging file [${source.getFile}] to [$connection]")
      val stagingDir: OverthereFile = connection.getFile(target.getStagingDirectoryPath)

      if (!stagingDir.exists()) {
        ctx.logOutput(s"Creating staging directory [$stagingDir]")
        stagingDir.mkdirs()
      }

      val taskStagingDir: OverthereFile = stagingDir.getFile(ctx.getTask.getId)
      if (!taskStagingDir.exists()) {
        ctx.logOutput(s"Creating staging directory for task [$taskStagingDir]")
        taskStagingDir.mkdir()
      }

      var counter = 0
      def tryCreateLoop(dir: OverthereFile): OverthereFile = Try({
        logger.debug(s"Attempting to create artifact staging dir [$dir]")
        dir.mkdir()
        dir
      }).getOrElse({
        counter += 1
        tryCreateLoop(taskStagingDir.getFile(s"${source.getName}-$counter"))
      })

      val artifactStagingDir: OverthereFile = tryCreateLoop(taskStagingDir.getFile(source.getName))
      ctx.logOutput(s"Successfully created artifact staging dir [$artifactStagingDir]")

      val stagedFile: OverthereFile = artifactStagingDir.getFile(source.getFile.getName)
      ctx.logOutput(s"Uploading [${source.getFile}] to [$stagedFile]")
      source.getFile.copyTo(stagedFile)
      logger.debug(s"Staged file [$stagedFile]")

      stagedFile
    }
  }

  private[stager] class _StagingContext extends StagingContext {
    val stagingSteps: ListBuffer[StagingStep] = new ListBuffer[StagingStep]()
    val cleanupHosts: mutable.Set[StagingTarget] = new mutable.HashSet[StagingTarget]()

    def stageArtifact(artifact: Artifact, target: StagingTarget): StagedFile = {
      if (Option(target.getStagingDirectoryPath).filterNot(_.trim.isEmpty).isEmpty) {
        new JustInTimeFile(artifact)
      } else {
        val file: StagingFile = new StagingFile(artifact, target)
        stagingSteps += new StagingStep(file)
        cleanupHosts += target
        file
      }
    }
  }

}
