package com.xebialabs.deployit.deployment.planner

import com.xebialabs.deployit.deployment.planner.PlanSugar._
import com.xebialabs.deployit.deployment.planner.StepPlan.StepWithPlanningInfo
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.services.Repository
import com.xebialabs.xlplatform.satellite.{Satellite, SatelliteAware}

import scala.collection.convert.wrapAll._
import scala.collection.mutable.ListBuffer

class SatelliteDeploymentPlanner(wrappedPlanner: Planner) extends Planner with PlanSugar {

  override def plan(spec: DeltaSpecification, repository: Repository): PhasedPlan = {
    val satellitePlan = wrappedPlanner.plan(spec, repository)
    val promotedPlan = promotePlan(satellitePlan)
    Satellites.prepareForSatelliteExecution(promotedPlan)
  }

  def promotePlan(satellitePlan: Plan): PhasedPlan = {
    satellitePlan match {
      case phasedPlan: PhasedPlan => promoteSatellite(phasedPlan)
      case planPhase: PlanPhase => new PhasedPlan(List(promoteSatellite(planPhase)), planPhase.getListeners)
      case plan: ExecutablePlan => new PhasedPlan(List(new PlanPhase(promoteSatellite(plan), plan.getDescription, plan.getListeners)), plan.getListeners)
    }
  }

  private def promoteSatellite(plan: ExecutablePlan): ExecutablePlan = {
    plan match {
      case stepPlan: StepPlan => promoteSatellite(stepPlan)
      case serialPlan: SerialPlan => promoteSatellite(serialPlan)
      case parallelPlan: ParallelPlan => promoteSatellite(parallelPlan)
    }
  }

  private def promoteSatellite(phasedPlan: PhasedPlan): PhasedPlan = {
    phasedPlan.copy(phases = phasedPlan.phases.map(promoteSatellite))
  }

  private def promoteSatellite(planPhase: PlanPhase): PlanPhase = {
    planPhase.copy(plan = promoteSatellite(planPhase.plan))
  }

  private def promoteSatellite(stepPlan: StepPlan): ExecutablePlan = {
    val result = stepPlan.getStepsWithPlanningInfo.foldLeft(List[(Satellite, ListBuffer[StepWithPlanningInfo])]()) {
      (tuples, step) =>

        val satellite = getSatelliteOrNull(step.getStep)

        tuples match {
          case Nil => List((satellite, ListBuffer(step)))
          case (currSat, buff) :: _ if currSat == satellite => buff.append(step)
            tuples
          case _ => (satellite, ListBuffer(step)) :: tuples
        }
    }
    buildStepPlanResult(result, stepPlan)
  }

  private def buildStepPlanResult(result: List[(Satellite, ListBuffer[StepWithPlanningInfo])], originalPlan: StepPlan): ExecutablePlan = {
    result match {
      case List((null, swpi)) => originalPlan
      case List((sat, swpi)) => originalPlan.copy(satellite = sat, description = originalPlan.getDescription + s" with satellite ${sat.getName}")
      case _ =>
        val reversedTuples: List[(Satellite, ListBuffer[StepWithPlanningInfo])] = result.reverse
        val stepPlans = reversedTuples.map {
          case (sat, swpi) if sat == null => originalPlan.copy(steps = swpi, satellite = sat)
          case (sat, swpi) => originalPlan.copy(originalPlan.getDescription + s" ${
            sat.getName
          }", swpi, findCheckpoints(swpi, originalPlan.getCheckpoints), satellite = sat)
        }

        originalPlan.toSerial(subPlans = stepPlans)
    }
  }

  private def promoteSatellite(plan: CompositePlan): ExecutablePlan = {
    val modifiedSubPlans = plan.getSubPlans.map(promoteSatellite)
    val satellites = modifiedSubPlans.map(_.satellite).toSet

    val promotedSatellite = if (satellites.size == 1) {
      satellites.head
    } else {
      null
    }

    plan.copy(subPlans = modifiedSubPlans, satellite = promotedSatellite)
  }

  private def getSatelliteOrNull(step: Step): Satellite = step match {
    case stepSat: SatelliteAware => stepSat.getSatellite
    case _ => null
  }
}
