package com.xebialabs.deployit.deployment.rules

import com.xebialabs.deployit.plugin.api.deployment.specification.Operation
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.rules.Scope
import com.xebialabs.platform.script.jython.ScriptSource
import grizzled.slf4j.Logging

import scala.xml.{Elem, Node, Text}
import RuleParser._

private[rules] class XmlRuleParser(ruleRegistry: RuleRegistry) extends Logging {

  private val ruleParsers = new ScriptRuleWithReferenceParser or new DeployedRuleParser or new PlanRuleParser

  def ruleFromXml(xml: Elem) {
    trace("Searching for <rule>")
    (xml \ "rule") foreach parseAndRegister
  }

  def disableRulesXml(xml: Elem) {
    trace("Searching for <disable-rule>")
    (xml \ "disable-rule") foreach disableRule
  }

  private def parseAndRegister(node: Node) = ruleParsers.parseRule(node) match {
    case Some(rule) => ruleRegistry.registerRule(rule)
    case None => warn(s"Could not parse ${node.text}")
  }

  private def disableRule(node: Node) = ruleRegistry.disableRule((node \ "@name").text)
}

private[rules] trait RuleParser {
  def parseRule(ruleNode: Node): Option[Rule]

  def ruleName(ruleNode: Node) = (ruleNode \ "@name").text

  def scope(ruleNode: Node) = Scope.getScope((ruleNode \ "@scope").text)
}

private[rules] class ScriptRuleWithReferenceParser extends ScriptParser {
  override def scriptElementName: String = "planning-script-path"

  override def scriptSource(scriptElement: Node): ScriptSource = ScriptSource.byResource(scriptElement text)
}

private[rules] trait ScriptParser extends RuleParser with ConditionsParser with ConditionValidation {
  override def parseRule(ruleNode: Node): Option[Rule] = scope(ruleNode) match {
    case Scope.DEPLOYED => (ruleNode \ scriptElementName).headOption.map { scriptElement =>
      ensureTypeAndOperationsPresent(ruleNode)
      ensureTypesAreDeployeds(ruleNode)
      new ScriptDeployedRule(ruleName(ruleNode), scriptSource(scriptElement), types(ruleNode), operations(ruleNode), condition(ruleNode))
    }
    case Scope.POST_PLAN | Scope.PRE_PLAN | Scope.PLAN => (ruleNode \ scriptElementName).headOption.map { scriptElement =>
      ensureTypeAndOperationNotPresent(ruleNode)
      new ScriptPlanRule(ruleName(ruleNode), scope(ruleNode), scriptSource(scriptElement), condition(ruleNode))
    }
  }

  def scriptElementName: String

  def scriptSource(scriptElement: Node): ScriptSource
}

private[rules] class DeployedRuleParser extends RuleParser with StepsParser with ConditionsParser with ConditionValidation with CheckpointValidation {
  override def parseRule(ruleNode: Node): Option[Rule] = scope(ruleNode) match {
    case Scope.DEPLOYED =>
      ensureTypeAndOperationsPresent(ruleNode)
      ensureCheckpointsAreValidForScopeDeployed(ruleNode)
      ensureTypesAreDeployeds(ruleNode)
      Option(new XmlDeployedRule(ruleName(ruleNode), types(ruleNode), operations(ruleNode), condition(ruleNode), steps(ruleNode)))
    case scope: Scope => None
  }
}

private[rules] class PlanRuleParser extends RuleParser with StepsParser with ConditionsParser with ConditionValidation with CheckpointValidation {
  override def parseRule(ruleNode: Node): Option[Rule] = scope(ruleNode) match {
    case Scope.POST_PLAN | Scope.PRE_PLAN | Scope.PLAN =>
      val ruleScope = scope(ruleNode)
      ensureTypeAndOperationNotPresent(ruleNode)
      ensureCheckpointAreValidForScopesPlan(ruleNode)
      Option(new XmlPlanRule(ruleName(ruleNode), ruleScope, condition(ruleNode), steps(ruleNode)))
    case scope: Scope => None
  }
}

private[rules] trait ConditionsParser {
  def condition(ruleNode: Node): Option[String] = (ruleNode \ "conditions" \ "expression").map(_.text).headOption

  def types(ruleNode: Node): Iterable[Type] = (ruleNode \ "conditions" \ "type").map(`type` => Type.valueOf(`type`.text))

  def operations(ruleNode: Node): Iterable[Operation] = (ruleNode \ "conditions" \ "operation").map(operation => Operation.valueOf(operation.text))
}

private[rules] trait StepsCollector {
  def collectFromSteps[T](ruleNode: Node, values: PartialFunction[Node, T]) = ruleNode \ "steps" flatMap (_.notTextChild) collect values
}

private[rules] trait StepsParser extends StepsCollector with StepParser with CheckpointParser {

  def steps(ruleNode: Node): Seq[XmlStepDataWithCheckPoint] = zipWithCheckpoint(collectFromSteps(ruleNode, checkpoint orElse step))

  def zipWithCheckpoint(stepNodes: Seq[XmlStepElem]): Seq[XmlStepDataWithCheckPoint] = stepNodes match {
    case Nil => Nil
    case (c: XmlCheckpoint) :: tail => throw new IllegalArgumentException("Checkpoint can not be first element in steps, and there can not be 2 checkpoints after each other")
    case (e: XmlStepData) :: (c: XmlCheckpoint) :: tail => XmlStepDataWithCheckPoint(e, Some(c)) +: zipWithCheckpoint(tail)
    case (e: XmlStepData) :: tail => XmlStepDataWithCheckPoint(e, None) +: zipWithCheckpoint(tail)
  }
}

private[rules] trait StepParser {

  def step: PartialFunction[Node, XmlStepElem] = {
    case step: Node => XmlStepData(step.label, step.notTextChild.map(parameter))
  }

  def parameter(parameter: Node) = parameter.notTextChild.isEmpty match {
    case false => mapParameter(parameter)
    case true => expressionOrBasicParameter(parameter)
  }

  def mapParameter(parameter: Node) = XmlCollectionParameter(parameter.label, parameter.notTextChild map expressionOrBasicParameter)

  def expressionOrBasicParameter(parameter: Node) = (parameter \ "@expression").headOption map (_.text.toBoolean) match {
    case Some(true) => XmlExpressionParameter(parameter.label, parameter.text.trim)
    case _ => XmlBasicParameter(parameter.label, parameter.text.trim)
  }
}

private[rules] trait CheckpointParser {
  def checkpoint: PartialFunction[Node, XmlCheckpoint] = {
    case checkpoint: Node if checkpoint.label == "checkpoint" =>
      XmlCheckpoint(
        (checkpoint \ "@name").headOption.map(_.text),
        (checkpoint \ "@completed").headOption.map(_.text).map(Operation.valueOf))
  }
}

private[rules] trait Validation {
  def ensurePresent[T](value: Iterable[T], name: String) = require(value.nonEmpty, s"At least one <$name> element has to be present")

  def ensureNotPresent[T](value: Iterable[T], name: String) = require(value.isEmpty, s"Element <$name> is not allowed here")
}

private[rules] trait ConditionValidation extends Validation {
  self: ConditionsParser =>

  private lazy val deployed: Type = Type.valueOf("udm.Deployed")

  def ensureTypeAndOperationsPresent(ruleNode: Node) {
    ensurePresent(types(ruleNode), "type")
    ensurePresent(operations(ruleNode), "operation")
  }

  def ensureTypeAndOperationNotPresent(ruleNode: Node) {
    ensureNotPresent(types(ruleNode), "type")
    ensureNotPresent(operations(ruleNode), "operation")
  }

  def ensureTypesAreDeployeds(ruleNode: Node) {
    types(ruleNode).foreach { t =>
      require(t.isSubTypeOf(deployed), s"Type [$t] is not a deployed type.")
    }
  }
}

private[rules] trait CheckpointValidation extends Validation with StepsCollector {
  self: CheckpointParser with ConditionsParser =>

  def ensureCheckpointsAreValidForScopeDeployed(ruleNode: Node) {
    val checkpoints = collectFromSteps(ruleNode, checkpoint)
    if (operations(ruleNode).toSet.forall(_ == Operation.MODIFY)) {
      ensureCheckpointsHaveCorrectOperation(checkpoints)
      val finalCheckpoints = checkpoints.filter(_.name.isEmpty)
      require(finalCheckpoints.size <= 2, "Can only have 2 'final' checkpoints, all other should contain a name")
      if (finalCheckpoints.size > 1) {
        require(finalCheckpoints.count(_.overrideOperation.isDefined) > 0, "Should have at least 1 'overrideOperation' when there are multiple 'final' checkpoints")
      }
    } else {
      require(checkpoints.forall(_.overrideOperation.isEmpty),
        "operation must be MODIFY and no other operation specified to be able to set override-operation on a checkpoint")
      if (checkpoints.size >= 1) {
        require(checkpoints.count(_.name.isEmpty) == 1, "Can only have 1 'final' checkpoint, all other should contain a name")
      }
    }
    if (checkpoints.size >= 1) {
      require(checkpoints.last.name.isEmpty, "The last checkpoint should not have a name, it should be 'final'")
    }
  }

  def ensureCheckpointAreValidForScopesPlan(ruleNode: Node) = ensureNotPresent(collectFromSteps(ruleNode, checkpoint), "checkpoint")

  private def ensureCheckpointsHaveCorrectOperation(checkpoints: Iterable[XmlCheckpoint]) = {
    val validOperations = Seq(Operation.CREATE, Operation.DESTROY, Operation.MODIFY)
    val valid = checkpoints.collect { case XmlCheckpoint(n, Some(o)) => o }.forall(validOperations.contains)
    require(valid, s"Checkpoint can have operation attribute set to ${validOperations.mkString(", ")} or not defined at all")
  }
}

private[rules] object RuleParser {

  implicit class RuleCombine(parser1: RuleParser) {
    def or(parser2: RuleParser): RuleParser = new CombinedRuleParser(parser1, parser2)
  }

  implicit class XmlUtils(node: Node) {
    def notTextChild = node.child.filterNot(_.isInstanceOf[Text])
  }

  private[RuleParser] class CombinedRuleParser(parser1: RuleParser, parser2: RuleParser) extends RuleParser {
    override def parseRule(ruleNode: Node): Option[Rule] = parser1.parseRule(ruleNode) match {
      case a: Some[Rule] => a
      case _ => parser2.parseRule(ruleNode)
    }
  }

}
