package com.xebialabs.deployit.deployment.planner;

import java.io.Serializable;
import java.util.List;
import javax.annotation.Nullable;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;

import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;

import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Iterables.isEmpty;
import static com.google.common.collect.Iterables.transform;
import static com.google.common.collect.Lists.newArrayList;
import static java.util.Collections.singletonList;
import static java.util.Collections.unmodifiableList;

public class MultiDeltaSpecification implements Serializable {
    private final DeltaSpecification mainDeltaSpecification;
    private final List<DeltaSpecification> deltaSpecifications;

    public MultiDeltaSpecification(final DeltaSpecification deltaSpecification) {
        this(deltaSpecification, singletonList(deltaSpecification));
    }

    private MultiDeltaSpecification(final DeltaSpecification mainDeltaSpecification, final Iterable<DeltaSpecification> deltaSpecifications) {
        Preconditions.checkArgument(!isEmpty(deltaSpecifications), "Wrong usage of internal API: deltaSpecifications must not be empty");
        Preconditions.checkNotNull(mainDeltaSpecification, "Wrong usage of internal API: mainDeltaSpecification must not be null");
        this.deltaSpecifications = newArrayList(deltaSpecifications);
        this.mainDeltaSpecification = mainDeltaSpecification;
    }

    public Operation getMainOperation() {
        return getMainDeltaSpecification().getOperation();
    }

    public DeployedApplication getMainDeployedApplication() {
        return getMainDeltaSpecification().getDeployedApplication();
    }

    public DeployedApplication getMainPreviousDeployedApplication() {
        return getMainDeltaSpecification().getPreviousDeployedApplication();
    }

    public DeltaSpecification getMainDeltaSpecification() {
        //To make old api happy, returns main delta specification (the one user choose to deploy) and ignoring dependencies atm
        //This should be changed when full multi-deployments come and all delta specification are treated equally
        return mainDeltaSpecification;
    }

    public List<DeltaSpecification> getDeltaSpecifications() {
        return unmodifiableList(deltaSpecifications);
    }

    public List<DeployedApplication> getAllDeployedApplications() {
        return newArrayList(transform(getDeltaSpecifications(), new Function<DeltaSpecification, DeployedApplication>() {
                    @Nullable
                    @Override
                    public DeployedApplication apply(final DeltaSpecification input) {
                        return input.getDeployedApplication();
                    }
                }
        ));
    }

    public List<Delta> getAllDeltas() {
        return newArrayList(concat(transform(getDeltaSpecifications(), new Function<DeltaSpecification, Iterable<Delta>>() {
                    @Nullable
                    @Override
                    public Iterable<Delta> apply(final DeltaSpecification input) {
                        return input.getDeltas();
                    }
                }
        )));
    }

    public static MultiDeltaSpecification withDependencies(DeltaSpecification main, List<DeltaSpecification> required){
        return new MultiDeltaSpecification(main, concat(required, singletonList(main)));
    }

    public static MultiDeltaSpecification forRollback(List<DeltaSpecification> deltaSpecifications) {
        return new MultiDeltaSpecification(deltaSpecifications.get(0), deltaSpecifications);
    }
}
