package com.xebialabs.deployit.deployment.planner

import com.xebialabs.deployit.deployment.planner.PlanSugar._
import com.xebialabs.deployit.deployment.planner.StepPlan.StepWithPlanningInfo
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification
import com.xebialabs.deployit.plugin.api.services.Repository

import scala.collection.convert.wrapAll._
import scala.collection.mutable

class OrderChunkingPlanner(wrappedPlanner: Planner) extends Planner with ChunkingPlannerBase with PlanSugar {

  val threshold: Int = 30

  val orderDescr = Map(0 -> "Pre flight", 1 -> "Stop artifacts", 2 -> "Stop containers", 3 -> "Undeploy artifacts", 4 -> "Destroy resources",
    5 -> "In between", 6 -> "Create resources", 7 -> "Deploy artifacts", 8 -> "Start containers", 9 -> "Start artifacts", 10 -> "Post flight")

  def plan(spec: DeltaSpecification, repository: Repository): PhasedPlan = {
    val complexPlan = wrappedPlanner.plan(spec, repository)
    chunkByOrder(complexPlan)
  }

  def chunkByOrder(complexPlan: Plan): PhasedPlan = {
    complexPlan match {
      case phasedPlan: PhasedPlan => chunkByOrder(phasedPlan)
      case planPhase: PlanPhase => new PhasedPlan(List(chunkByOrder(planPhase)), planPhase.getListeners)
      case plan: ExecutablePlan => new PhasedPlan(List(new PlanPhase(chunkByOrder(plan), plan.getDescription, plan.getListeners)), plan.getListeners)
    }
  }

  def chunkByOrder(plan: PhasedPlan): PhasedPlan = {
    plan.copy(phases = plan.phases.map(chunkByOrder))
  }

  def chunkByOrder(planPhase: PlanPhase): PlanPhase = {
    planPhase.copy(plan = chunkByOrder(planPhase.plan))
  }

  def chunkByOrder(plan: ExecutablePlan): ExecutablePlan = plan match {
    case cp: CompositePlan => cp.copy(subPlans = cp.getSubPlans.map(chunkByOrder))
    case stp: StepPlan if stp.getSteps.size() > threshold => val chunks = chunkByOrder(stp)
      if (chunks.length <= 1) stp
      else stp.toSerial(
        subPlans = chunks)
    case stp: StepPlan => stp
  }

  def chunkByOrder(plan: StepPlan): List[StepPlan] = {
    plan.getStepsWithPlanningInfo
      .groupBy(_.getStep.getOrder / 10).toList
      .sortBy(_._1)
      .map(createStepPlanForOrderedChunks(_, plan))
  }

  def createStepPlanForOrderedChunks(b: (Int, mutable.Buffer[StepWithPlanningInfo]), plan: StepPlan): StepPlan = {
    b match {
      case (i, ss) => plan.copy(s"Steps with order ${orderDescr.getOrElse(i, "Undefined")}", ss, findCheckpoints(ss, plan.getCheckpoints.toList))
    }
  }

}
