package com.xebialabs.deployit.deployment
package orchestrator

import java.util

import com.xebialabs.deployit.deployment.orchestrator.DescriptionHelper._
import com.xebialabs.deployit.deployment.rules.PlannerException
import com.xebialabs.deployit.engine.spi.orchestration.Orchestrations._
import com.xebialabs.deployit.engine.spi.orchestration.Orchestrator
import com.xebialabs.deployit.plugin.api.deployment.specification.{Delta, DeltaSpecification}
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.xlplatform._

import scala.collection.convert.wrapAll._
import scala.collection.mutable

trait PackageOrchestratorBase extends OrchestratorUtil {

  def matchDeployableIdByPackageId(delta: Delta, version: Version) =
    delta.correctDeployed.getDeployable.asInstanceOf[Deployable].getId.startsWith(version.getId)

  def getOrchestrations(spec: DeltaSpecification) = spec.getDeployedApplication.getVersion match {
    case comp: CompositePackage if (hasMatchingDeployed(spec)) =>
      val orderForOperation = getOrderForInt(spec.getOperation)
      val previousVersions: util.List[Version] = Option(spec.getPreviousDeployedApplication).map(_.getVersion.asInstanceOf[CompositePackage].getPackages).getOrElse(new util.ArrayList[Version]())
      val indexedPackages: List[(Version, Int)] = (getAllDeploymentPackages(comp) ++ previousVersions).toList.zipWithIndex
      val versionToDeltas: Map[(Version, Int), mutable.Buffer[Delta]] =
        spec.getDeltas.groupBy(delta => {
          val matchedPackage = indexedPackages.find(version => matchDeployableIdByPackageId(delta, version._1))
          matchedPackage.getOrElse(deployableNotFound(delta, indexedPackages.map(_._1).toSet))
        })

      versionToDeltas.toList.sortBy(_._1._2)(orderForOperation)
        .map { case (v, ds) => interleaved(getDescriptionForPackage(spec.getOperation, v._1), ds) }
    case _ => List(new DefaultOrchestrator().orchestrate(spec))
  }

  private def hasMatchingDeployed(spec: DeltaSpecification): Boolean = {
    val deployeds = spec.getDeltas.map(delta => delta.correctDeployed)
    deployeds.filter(p => spec.getDeployedApplication.getDeployeds.contains(p)).nonEmpty

  }

  private def deployableNotFound(delta: Delta, versions: Set[Version]) = {
    val deployableId = delta.correctDeployed.getDeployable.asInstanceOf[Deployable].getId
    val pkgs = versions.mkString(", ")
    throw new PlannerException(s"Deployable [${deployableId}] not found in any deployment package referenced by the composite [${pkgs}].")
  }

  private def getAllDeploymentPackages(comp: CompositePackage): Seq[Version] = {
    comp.getPackages.flatMap {
      case depl: DeploymentPackage => List(depl)
      case comp: CompositePackage => getAllDeploymentPackages(comp)
    }
  }
}

@Orchestrator.Metadata(name = "sequential-by-composite-package", description = "The sequential by composite package orchestrator")
class SequentialByPackageOrchestrator extends Orchestrator with PackageOrchestratorBase {
  def orchestrate(spec: DeltaSpecification) = serial(getDescriptionForSpec(spec), getOrchestrations(spec))
}

@Orchestrator.Metadata(name = "parallel-by-composite-package", description = "The parallel by composite package orchestrator")
class ParallelByPackageOrchestrator extends Orchestrator with PackageOrchestratorBase {
  def orchestrate(spec: DeltaSpecification) = parallel(getDescriptionForSpec(spec), getOrchestrations(spec))
}
