package com.xebialabs.deployit.deployment.rules

import java.util

import com.github.drapostolos.typeparser.TypeParser
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.platform.script.jython.{JythonContext, JythonSupport}

class XmlParameterResolver(stepRegistry: StepRegistry) extends JythonSupport {

  private val parser: TypeParser = TypeParser.newBuilder().build()

  def resolveXmlParameters(stepName: String, parameters: Iterable[XmlParameter])(implicit jythonContext: JythonContext): Map[String, Any] = {
    implicit val stepDescriptor: StepDescriptor = stepRegistry.getStepDescriptor(stepName)
    parameters.map(p => p.name -> resolveXmlParameter(p)).toMap
  }

  private def resolveXmlParameter(parameter: XmlParameter)(implicit stepDescriptor: StepDescriptor, jythonContext: JythonContext) = parameter match {
    case basic: XmlBasicParameter => resolveBasicParameter(basic)
    case expression: XmlExpressionParameter => resolveExpressionParameter(expression)
    case collection: XmlCollectionParameter =>
      val x: Option[(String, StepParameterDescriptor)] = stepDescriptor.parameters.find(_._1 == collection.name)
      x match {
        case Some((name, param)) if param.isFieldOfType[util.Map[_, _]] => resolveMapParameter(collection)
        case Some((name, param)) if param.isFieldOfType[util.List[_]] => resolveListParameter(collection)
        case _ => {
          throw new IllegalArgumentException(s"Cannot resolve parameter ${collection.name} as a collection")
        }
      }
    case ci: XmlConfigurationItemParameter => resolveCiParameter(ci)
  }


  private def resolveCiParameter(parameter: XmlConfigurationItemParameter)(implicit stepDescriptor: StepDescriptor, jythonContext: JythonContext): Any = {
    val descriptor = DescriptorRegistry.getDescriptor(parameter.ciType)
    val fieldValues = parameter.fieldValues
    val ciId = evaluateField(fieldValues.get("id").get).asInstanceOf[String]
    fieldValues.filter(fieldValue => fieldValue._1 != "id").foldLeft(descriptor.newInstance(ciId))((ci: ConfigurationItem, fieldValue) => {
      fieldValue._1 match {
        case name if descriptor.getPropertyDescriptor(name) != null => ci.setProperty(fieldValue._1, evaluateField(fieldValue._2))
        case invalidParameter => throw new IllegalArgumentException(s"step contains an invalid parameter '$invalidParameter' not present in ci with id '$ciId'")
      }
      ci
    })
  }

  private def evaluateField(parameter: XmlConfigurationItemFieldParameter)(implicit stepDescriptor: StepDescriptor, jythonContext: JythonContext): Any = {
    parameter match {
      case XmlConfigurationItemFieldParameter(name, expression, true) => evaluateExpression[Any](expression, ProxyUnwrapper.unwrap)
      case XmlConfigurationItemFieldParameter(name, value, false) => value
    }
  }

  private def resolveBasicParameter(parameter: XmlBasicParameter)(implicit stepDescriptor: StepDescriptor, jythonContext: JythonContext): Any = {
    val valueType = stepDescriptor.parameters.get(parameter.name).map(_.parameterType.fieldType).getOrElse(classOf[String])
    Option(parameter).map(_.value).filter(_.nonEmpty).map(value => parseValueAsType(value, valueType)).orNull
  }

  private def parseValueAsType(value: String, valueType: Class[_]) = {
    require(parser.isTargetTypeParsable(valueType), s"Value '$value' cannot be parsed as type ${valueType.getName}")
    parser.parse(value, valueType)
  }

  private def resolveExpressionParameter(parameter: XmlExpressionParameter)(implicit stepDescriptor: StepDescriptor, jythonContext: JythonContext) = {
    evaluateExpression[Any](parameter.expression, ProxyUnwrapper.unwrap)
  }

  private def resolveMapParameter(map: XmlCollectionParameter)(implicit stepDescriptor: StepDescriptor, jythonContext: JythonContext): Map[String, Any] = {
    map.parameters.map(p => p.name -> resolveXmlParameter(p)).toMap
  }

  private def resolveListParameter(list: XmlCollectionParameter)(implicit stepDescriptor: StepDescriptor, jythonContext: JythonContext): Seq[Any] = {
    require(list.parameters.map(_.name).toSet.size <= 1, "List parameter can have only value tag specified")
    require(list.parameters.map(_.name).toSet.headOption.getOrElse("value") == "value", "List parameter can have only value tag specified")
    list.parameters.map(resolveXmlParameter).toSeq
  }
}

object ProxyUnwrapper {
  private[rules] val unwrap: JythonSupport.PreprocessExpression = exp => s"_unwrapper.unwrap($exp)"
}
