package com.xebialabs.deployit.deployment
package rules

import com.xebialabs.deployit.plugin.api.flow.Step
import nl.javadude.scannit.Scannit
import collection.convert.wrapAll._
import com.xebialabs.deployit.plugin.api.rules.{StepMetadata, StepParameter}
import java.lang.reflect.Field
import com.xebialabs.xlplatform._


object StepClassScanner {
  def fill(stepRegistry: StepRegistry, scannit: Scannit = Scannit.getInstance, errorStrategy: List[String] => Unit = JavaBasedRuleBuilder.throwStrategy) {
    new StepClassScanner(stepRegistry, scannit).registerAllSteps(errorStrategy)
  }
}

class StepClassScanner(stepRegistry: StepRegistry, scannit: Scannit) {

  def registerAllSteps(errorStrategy: List[String] => Unit) {
    val classes = scannit.getTypesAnnotatedWith(classOf[StepMetadata])
    var errMsgs = List[String]()
    for (c <- classes) {
      var registerThisClass = true
      if (!classOf[Step].isAssignableFrom(c)) {
        errMsgs +:= s"Step class ${c.getName} with StepMetaData should be instance of Step"
        registerThisClass = false
      }
      if (!c.getConstructors.exists(_.getParameterTypes.size == 0)) {
        errMsgs +:= s"Step class ${c.getName} should have a default constructor"
        registerThisClass = false
      }
      if (registerThisClass) {
        val params = getStepParameterDescriptors(c)
        val stepDescriptor = StepPrimitiveDescriptor(c.getAnnotation(classOf[StepMetadata]).name(), c.asInstanceOf[Class[_ <: Step]], params)
        stepRegistry.register(stepDescriptor)
      }
    }
    errorStrategy(errMsgs)
  }

  private def getStepParameterDescriptors(clazz: Class[_]): Map[String, StepPrimitiveParameterDescriptor] = {
    getClassChain(clazz).flatMap(_.getDeclaredFields).collect {
      case f if Option(f.getAnnotation(classOf[StepParameter])).isDefined => (name(f).camelCaseToDashCase, createStepParameterDescriptor(f))
    }.toMap
  }

  private def name(field: Field) = {
    val alternativeName = field.getAnnotation(classOf[StepParameter]).name()
    if(alternativeName.isEmpty) field.getName else alternativeName
  }

  private def createStepParameterDescriptor(field: Field) = {
    val stepParameter = field.getAnnotation(classOf[StepParameter])
    StepPrimitiveParameterDescriptor(stepParameter.description(), field, stepParameter.required())
  }
}
