package com.xebialabs.deployit.deployment.rules

import java.lang.Iterable
import java.util

import com.xebialabs.deployit.deployment.planner.{DefaultDeploymentPlanningContext, StepPlan}
import com.xebialabs.deployit.deployment.rules.RulePlanningContext._
import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentStep
import com.xebialabs.deployit.plugin.api.deployment.planning.{Checkpoint, DeploymentPlanningContext}
import com.xebialabs.deployit.plugin.api.deployment.specification.{Delta, DeltaSpecification, Operation}
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.services.Repository
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication
import com.xebialabs.deployit.plugin.api.xld.AppliedDistribution

object RulePlanningContext {
  def apply(spec: DeltaSpecification, stepRegistry: StepRegistry, readOnlyRepository: Repository, plan: StepPlan) = {
    new RulePlanningContext(spec, readOnlyRepository, plan) {
      override def getStepRegistry: StepRegistry = stepRegistry
    }
  }

  private[RulePlanningContext] def whichDeployedApplication(deltaSpecification: DeltaSpecification): AppliedDistribution = deltaSpecification.getOperation match {
    case Operation.DESTROY => null
    case _ => deltaSpecification.getAppliedDistribution
  }
}

sealed abstract class RulePlanningContext(deltaSpecification: DeltaSpecification,
                                          readOnlyRepository: Repository,
                                          plan: StepPlan,
                                          wrappedContext: DeploymentPlanningContext,
                                          backwardsCompatible: Boolean = false) extends ScalaJavaBridgeDeploymentPlanningContext {


  def this(deltaSpecification: DeltaSpecification, repo: Repository, plan: StepPlan) =
    this(deltaSpecification, repo, plan,
      new DefaultDeploymentPlanningContext(whichDeployedApplication(deltaSpecification), deltaSpecification.getPreviousAppliedDistribution, repo, plan, deltaSpecification.isRollback()))

  def getStepRegistry: StepRegistry

  def backwardCompatible: RulePlanningContext = {
    new RulePlanningContext(deltaSpecification, readOnlyRepository, plan, wrappedContext, true) {
      override def getStepRegistry: StepRegistry = RulePlanningContext.this.getStepRegistry
    }
  }

  override def getDeployedApplication: DeployedApplication = deltaSpecification.getOperation match {
    case Operation.DESTROY if backwardsCompatible => getPreviousDeployedApplication
    case _ => wrappedContext.getDeployedApplication
  }

  override def getAppliedDistribution: AppliedDistribution = deltaSpecification.getOperation match {
    case Operation.DESTROY if backwardsCompatible => getPreviousAppliedDistribution
    case _ => wrappedContext.getAppliedDistribution
  }

  override def addStepWithCheckpoint(step: Step, checkpoint: Checkpoint): Unit = wrappedContext.addStepWithCheckpoint(step, checkpoint)

  override def addStepWithCheckpoint(step: Step, delta: Delta): Unit = wrappedContext.addStepWithCheckpoint(step, delta)

  override def addStepWithCheckpoint(step: Step, delta: Delta, overrideOperation: Operation): Unit = wrappedContext.addStepWithCheckpoint(step, delta, overrideOperation)

  override def addStepWithCheckpoint(step: Step, deltas: Iterable[Delta]): Unit = wrappedContext.addStepWithCheckpoint(step, deltas)

  override def getAttribute(name: String): AnyRef = wrappedContext.getAttribute(name)

  override def getPreviousDeployedApplication: DeployedApplication = wrappedContext.getPreviousDeployedApplication

  override def getPreviousAppliedDistribution: AppliedDistribution = wrappedContext.getPreviousAppliedDistribution

  override def setAttribute(name: String, value: scala.Any): Unit = wrappedContext.setAttribute(name, value)

  override def addCheckpoint(step: Step, checkpoint: Checkpoint): Unit = wrappedContext.addCheckpoint(step, checkpoint)

  override def addCheckpoint(step: Step, delta: Delta): Unit = wrappedContext.addCheckpoint(step, delta)

  override def addCheckpoint(step: Step, delta: Delta, overrideOperation: Operation): Unit = wrappedContext.addCheckpoint(step, delta, overrideOperation)

  override def addCheckpoint(step: Step, deltas: Iterable[Delta]): Unit = wrappedContext.addCheckpoint(step, deltas)

  override def addSteps(steps: util.Collection[DeploymentStep]): Unit = wrappedContext.addSteps(steps)

  override def addSteps(steps: Iterable[Step]): Unit = wrappedContext.addSteps(steps)

  override def getRepository: Repository = wrappedContext.getRepository

  override def addStep(step: DeploymentStep): Unit = wrappedContext.addStep(step)

  override def addStep(step: Step): Unit = wrappedContext.addStep(step)

  override def isRollback: Boolean = wrappedContext.isRollback
}