package com.xebialabs.platform.watcher

import akka.actor._
import com.xebialabs.platform.watcher.FileWatchActor.Messages._
import com.xebialabs.platform.watcher.FileWatchActor.Messages.Modified
import com.xebialabs.platform.watcher.FileWatchActor.Messages.Watch
import com.xebialabs.platform.settings.Settings
import scalax.file.Path


object FileWatchActor {

  def props = Props(new FileWatchActor)

  object Messages {

    case class Watch(path: Path)
    case class StartWatching(path: Path)
    case class WatchStopped(path: Path)
    case class Poll()

    sealed trait SystemEvent
    case class Created(path: Path) extends SystemEvent
    case class Modified(path: Path) extends SystemEvent
    case class Deleted(path: Path) extends SystemEvent
  }
}

class FileWatchActor extends Actor with ActorLogging with Stash {

  private var subscriptions = List[Subscription]()

  override def preStart() {
    if (Settings(context).FileWatch.pollingEnabled) {
      implicit val dispatcher = context.system.dispatcher
      log.info(s"File changes polling is enabled")
      val interval = Settings(context).FileWatch.interval
      context.system.scheduler.schedule(interval, interval, self, Poll())
    } else {
      log.warning("File changes polling is disabled")
    }
  }

  override def postStop() {
    subscriptions.foreach(_.stopPolling())
    subscriptions = Nil
  }

  override def receive = {
    case Watch(path) => startWatch(path)
    case Poll() => pollFileEvents()
  }

  private def startWatch(path: Path) = {
    log.debug(s"Watching changes on path ${path.path}")
    val subscription = new Subscription(sender(), path)
    subscriptions = subscription +: subscriptions
    sender() ! StartWatching(path)
  }

  private def pollFileEvents() {
    subscriptions.foreach(_.notifyIfFileModified())
  }
}

private[watcher] class Subscription(notifyActor: ActorRef, fileToWatch: Path) extends FileWatch(fileToWatch) {

  def notifyIfFileModified() = doWhenModified {
    case StateChange.Created => notifyActor ! Created(fileToWatch)
    case StateChange.Modified => notifyActor ! Modified(fileToWatch)
    case StateChange.Deleted => notifyActor ! Deleted(fileToWatch)
  }

  def stopPolling() = notifyActor ! WatchStopped(fileToWatch)
}
