package com.xebialabs.deployit.booter.remote;

import org.apache.http.HttpHost;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.AuthCache;
import org.apache.http.client.HttpClient;
import org.apache.http.client.protocol.ClientContext;
import org.apache.http.conn.params.ConnRoutePNames;
import org.apache.http.conn.scheme.Scheme;
import org.apache.http.conn.ssl.SSLSocketFactory;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.BasicAuthCache;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.conn.PoolingClientConnectionManager;
import org.apache.http.impl.conn.tsccm.ThreadSafeClientConnManager;
import org.apache.http.params.CoreConnectionPNames;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.atomic.AtomicReference;

public class HttpClientHolder {
    String username;
    String password;

    DefaultHttpClient httpClient;
    HttpHost host;
    HttpHost proxyHost;
    BasicHttpContext localcontext;

    HttpClientHolder(BooterConfig config) {
        PoolingClientConnectionManager connManager = new PoolingClientConnectionManager();
        httpClient = new DefaultHttpClient(connManager);
        if (config.isSecure()) {
            setupSecureComms();
        }
        host = new HttpHost(config.getHost(), config.getPort(), String.format("http%s", config.isSecure() ? "s" : ""));
        localcontext = new BasicHttpContext();
        username = config.getUsername();
        password = config.getPassword();
        addCredentials();
        enablePreemptiveAuthentication();
        httpClient.getParams().setIntParameter(CoreConnectionPNames.CONNECTION_TIMEOUT, 10000);
        if(config.isProxied()) {
            setupHttpProxy(config);
        }
    }

    private void setupSecureComms() {
        try {
            SSLSocketFactory sslSocketFactory = new SSLSocketFactory(new TrustSelfSignedStrategy());
            Scheme scheme = new Scheme("https", 443, sslSocketFactory);
            httpClient.getConnectionManager().getSchemeRegistry().register(scheme);
        } catch (Exception e) {
            throw new IllegalStateException("Cannot setup secure communications with XL Deploy", e);
        }
    }

    private void setupHttpProxy(BooterConfig config) {
        proxyHost = new HttpHost(config.getProxyHost(), config.getProxyPort());
        logger.debug("Configuring connection via proxy: {}", proxyHost);
        httpClient.getParams().setParameter(ConnRoutePNames.DEFAULT_PROXY, proxyHost);
    }

    void shutdown() {
        getHttpClient().getConnectionManager().shutdown();
    }

    private void addCredentials() {
        final UsernamePasswordCredentials credentials = new UsernamePasswordCredentials(username, password);
        httpClient.getCredentialsProvider().setCredentials(AuthScope.ANY, credentials);
    }

    private void enablePreemptiveAuthentication() {
        AuthCache authCache = new BasicAuthCache();
        BasicScheme basicAuth = new BasicScheme();
        authCache.put(host, basicAuth);
        localcontext.setAttribute(ClientContext.AUTH_CACHE, authCache);
    }

    public void logout() {
        this.username = null;
        this.password = null;
        httpClient.getCookieStore().clear();
        httpClient.getCredentialsProvider().clear();
    }

    public String getUserName() {
        return this.username;
    }

    public void loginAs(String username, String password) {
        if (httpClient.getCredentialsProvider().getCredentials(AuthScope.ANY) != null) {
            logger.error("You're still logged in as another user, please logout first.");
            return;
        }

        this.username = username;
        this.password = password;

        addCredentials();
        httpClient.getCookieStore().clear();
    }


    public HttpClient getHttpClient() {
        return httpClient;
    }

    public HttpContext getHttpContext() {
        return localcontext;
    }

    public HttpHost getHost() {
        return host;
    }

    public HttpHost getProxyHost() {
        return proxyHost;
    }

    private static final Logger logger = LoggerFactory.getLogger(HttpClientHolder.class);
}