package com.xebialabs.deployit.booter.remote;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistryId;
import com.xebialabs.deployit.plugin.api.reflect.Type;

import static com.google.common.base.Preconditions.checkArgument;

public class RemoteDescriptorRegistry extends DescriptorRegistry {

    private final ReentrantReadWriteLock readWriteLock = new ReentrantReadWriteLock(true);
    private final Lock readLock = readWriteLock.readLock();
    private final Lock writeLock = readWriteLock.writeLock();

    private Map<Type, Descriptor> descriptors = Maps.newHashMap();
    private Multimap<Type, Type> subtypes = HashMultimap.create();

    protected RemoteDescriptorRegistry(DescriptorRegistryId id) {
        super(id);
    }

    @Override
    protected boolean isLocal() {
        return false;
    }

    @Override
    protected Collection<Descriptor> _getDescriptors() {
        readLock.lock();
        try {
            return descriptors.values();
        } finally {
            readLock.unlock();
        }
    }

    @Override
    protected Collection<Type> _getSubtypes(Type supertype) {
        readLock.lock();
        try {
            return subtypes.get(supertype);
        } finally {
            readLock.unlock();
        }
    }

    @Override
    protected Descriptor _getDescriptor(Type type) {
        checkArgument(descriptors.containsKey(type), "DescriptorRegistry does not know about type [%s]", type);
        readLock.lock();
        try {
            return descriptors.get(type);
        } finally {
            readLock.unlock();
        }
    }

    @Override
    protected boolean _exists(Type type) {
        readLock.lock();
        try {
            return descriptors.containsKey(type);
        } finally {
            readLock.unlock();
        }
    }

    public static void boot(DeployitCommunicator communicator) {
        RemoteDescriptorRegistry registry = new RemoteDescriptorRegistry(communicator.getConfig());
        DescriptorRegistry.add(registry);
        registry.reboot(communicator);
    }

    public void reboot(DeployitCommunicator communicator) {
        writeLock.lock();
        try {
            List<Descriptor> list = communicator.getProxies().getMetadataService().listDescriptors();
            reboot(list);
        } finally {
            writeLock.unlock();
        }
    }

    public void reboot(List<Descriptor> list) {
        writeLock.lock();
        try {
            Map<Type, Descriptor> newDescriptorsMap = Maps.newHashMap();
            Multimap<Type, Type> newSubtypesMap = HashMultimap.create();

            for (Descriptor descriptor : list) {
                newDescriptorsMap.put(descriptor.getType(), descriptor);
                for (Type type : descriptor.getSuperClasses()) {
                    newSubtypesMap.put(type, descriptor.getType());
                }
                for (Type type : descriptor.getInterfaces()) {
                    newSubtypesMap.put(type, descriptor.getType());
                }
            }

            this.descriptors = newDescriptorsMap;
            this.subtypes = newSubtypesMap;
        } finally {
            writeLock.unlock();
        }
    }

    public Collection<Descriptor> getLoadedDescriptors() {
        return _getDescriptors();
    }

    public Descriptor getLoadedDescriptor(Type type) {
        return _getDescriptor(type);
    }

    public Descriptor getLoadedDescriptor(String prefixedName) {
        return getLoadedDescriptor(lookupType(prefixedName));
    }

    public Descriptor getLoadedDescriptor(String prefix, String name) {
        return getLoadedDescriptor(lookupType(prefix, name));
    }
}